#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/io.h>
#include <linux/dma-mapping.h>

#include "dmw96ciu_regs.h"
#include <media/dmw96ciu.h>
#include <mach/platform.h>
#include <mach/hardware.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/clk.h>
#include <linux/slab.h>
#include <linux/dma-mapping.h>
#include <linux/vmalloc.h>
#include <linux/platform_device.h>
#include <linux/dmw96ciu_common.h>
#include <media/v4l2-common.h>
#include <media/v4l2-ioctl.h>
#include <media/v4l2-dev.h>
#include <media/videobuf-dma-contig.h>


void __iomem *dmw96ciu_regs;

/* CIU Crop capabilities */
struct ciu_device *dmw96ciu;
static struct device *ciu_dev;

/*For statistics*/
static unsigned int fifo_full_cnt = 0;

/* parallel/MIPI selection */
static enum
{
	CIU_INTERFACE_PARALLEL,
	CIU_INTERFACE_MIPI,
} ciu_interface_type = CIU_INTERFACE_PARALLEL;

/* List of image formats supported via CIU */
const static struct v4l2_fmtdesc ciu_formats[] = {
    {
		/* Note:  V4L2 defines FMT_NV16 as:
		*	Two planes -- one Y, one Cb + Cr interleaved
		*   y0y1y2y3 u0v0u1v1
		*/
		.description    = "Y/CbCr, 4:2:2",
		.pixelformat    = V4L2_PIX_FMT_NV16,
	},
	{
		/* Note:  V4L2 defines FMT_NV61 as:
		*	Two planes -- one Y, one Cb + Cr interleaved
		*   y0y1y2y3 v0u0v1u1
		*/
		.description    = "Y/CrCb, 4:2:2",
		.pixelformat    = V4L2_PIX_FMT_NV61,
	},
	{
	   /* Note:  V4L2 defines FMT_NV12 as:
		*	Two planes -- one Y, one Cb + Cr interleaved
		*   y0y1y2y3 u0v0
		*/
	   .description    = "Y/CbCr, 4:2:0",
	   .pixelformat    = V4L2_PIX_FMT_NV12,
	},
	{
	   /* Note:  V4L2 defines FMT_NV12 as:
		*	Two planes -- one Y, one Cb + Cr interleaved
		*   y0y1y2y3 v0u0
		*/
	   .description    = "Y/CrCb, 4:2:0",
	   .pixelformat    = V4L2_PIX_FMT_NV21,
	},
};

enum pixel_format_ciu {
	CIU_V4L2_PIX_FMT_NV16,
	CIU_V4L2_PIX_FMT_NV61,
	CIU_V4L2_PIX_FMT_NV12,
	CIU_V4L2_PIX_FMT_NV21,
};

struct ciu_buf {
	void (*complete)(struct videobuf_buffer *vb, void *priv);
	struct videobuf_buffer *vb;
	void *priv;
};

#define NUM_BUFS		(NUM_SW_BUFS + 1)

#define CIU_BUFS_IS_FULL(bufs)					\
	(((bufs)->queue + 1) % NUM_BUFS == (bufs)->done)
#define CIU_BUFS_IS_EMPTY(bufs)		((bufs)->queue == (bufs)->done)
#define CIU_BUFS_IS_LAST(bufs)					\
	((bufs)->queue == ((bufs)->done + 1) % NUM_BUFS)
#define CIU_BUFS_QUEUED(bufs)						\
	( (bufs)->queue >= (bufs)->done ) ? ((bufs)->queue - (bufs)->done) : ((((bufs)->done - (bufs)->queue + NUM_BUFS)) % NUM_BUFS)
#define CIU_BUF_DONE(bufs)		((bufs)->buf + (bufs)->done)
#define CIU_BUF_NEXT_DONE(bufs)				\
	((bufs)->buf + ((bufs)->done + 1) % NUM_BUFS)
#define CIU_BUF_QUEUE(bufs)		((bufs)->buf + (bufs)->queue)
#define CIU_BUF_MARK_DONE(bufs)				\
	(bufs)->done = ((bufs)->done + 1) % NUM_BUFS;
#define CIU_BUF_MARK_QUEUE(bufs)			\
	(bufs)->queue = ((bufs)->queue + 1) % NUM_BUFS;

struct ciu_bufs {

	spinlock_t lock;	/* For handling current buffer */
	/* queue full: (ciug.queue + 1) % NUM_BUFS == ciug.done
	   queue empty: ciug.queue == ciug.done */
	struct ciu_buf buf[NUM_BUFS];
	/* Next slot to queue a buffer. */
	int queue;
	/* Buffer that is being processed. */
	int done;
};


/**
 * struct ciu - Structure for storing CIU Control module 
 * information 
 * @lock: Spinlock to sync between isr and processes.
 * @ciu_mutex: Semaphore used to get access to the CIU.
 * @ref_count: Reference counter.
 * @ciu_clk: Pointer to CIU Interface clock.
 *
 * This structure is used to store the CIU Control Information.
 */
static struct ciu {
	spinlock_t lock;	/* For handling registered ciu callbacks */
	struct mutex ciu_mutex;	/* For handling ref_count field */
	int ref_count;
	struct clk *ciu_clk;
	struct ciu_pdata *pdata;
	dma_addr_t tmp_buf;
	size_t tmp_buf_size;
	unsigned long tmp_buf_offset;

	dma_addr_t dummy_buf;
	size_t dummy_buf_size;
	void* dummy_buf_virt;

	struct ciu_bufs bufs;
	struct ciu_bufs hw_bufs;

	struct v4l2_pix_format pix;

	unsigned int ciu_hw_index;
	unsigned int ciu_scaler_mode;
	/*Need for PIP*/
	struct v4l2_rect pip_crect;

} ciu_obj;

#define CIU_NUM_OF_INDEX 4
#define CIU_TYPE_OF_REGS 4
#define CIU_TYPE_OF_SCALE 3
enum { CIU_Y_L = 0 , CIU_Y_H , CIU_UV_L , CIU_UV_H };

unsigned int ciu_dma_address[CIU_NUM_OF_INDEX][CIU_TYPE_OF_REGS][CIU_TYPE_OF_SCALE] =
{ { {YDMA_ADDR1L, SYDMA_ADDR1L, PYDMA_ADDR1L } , {YDMA_ADDR1H, SYDMA_ADDR1H, PYDMA_ADDR1H} , {UDMA_ADDR1L, SUDMA_ADDR1L,PUDMA_ADDR1L} , {UDMA_ADDR1H, SUDMA_ADDR1H, PUDMA_ADDR1H} } , //index1
  { {YDMA_ADDR2L, SYDMA_ADDR2L, PYDMA_ADDR2L } , {YDMA_ADDR2H, SYDMA_ADDR2H, PYDMA_ADDR2H} , {UDMA_ADDR2L, SUDMA_ADDR2L,PUDMA_ADDR2L} , {UDMA_ADDR2H, SUDMA_ADDR2H, PUDMA_ADDR2H} } , //index2
  { {YDMA_ADDR3L, SYDMA_ADDR3L, PYDMA_ADDR3L } , {YDMA_ADDR3H, SYDMA_ADDR3H, PYDMA_ADDR3H} , {UDMA_ADDR3L, SUDMA_ADDR3L,PUDMA_ADDR3L} , {UDMA_ADDR3H, SUDMA_ADDR3H, PUDMA_ADDR3H} } , //index3
  { {YDMA_ADDR4L, SYDMA_ADDR4L, PYDMA_ADDR4L } , {YDMA_ADDR4H, SYDMA_ADDR4H, PYDMA_ADDR4H} , {UDMA_ADDR4L, SUDMA_ADDR4L,PUDMA_ADDR4L} , {UDMA_ADDR4H, SUDMA_ADDR4H, PUDMA_ADDR4H} } , //index3
};


unsigned int bytes_per_line(unsigned int width, unsigned int pixelformat ){

	return ((width * ( (pixelformat == V4L2_PIX_FMT_NV12 || pixelformat == V4L2_PIX_FMT_NV21) ? 6 : 8)) / 4);
}

unsigned int image_size(unsigned int width, unsigned int height, unsigned int pixelformat ){

	return (PAGE_ALIGN((width * ( (pixelformat == V4L2_PIX_FMT_NV12 || pixelformat == V4L2_PIX_FMT_NV21) ? 6 : 8) * height ) / 4));
}


int ciu_switch_addreses(unsigned int index, unsigned int buff_addr, unsigned int buff_virt_addr ,unsigned int xsize, unsigned int ysize, unsigned int type)
{
    unsigned int luma, chroma, header;
	unsigned int y_add_l=0 , y_add_h=0 , uv_add_l=0 , uv_add_h = 0;
	unsigned int* header_p;
	unsigned int header_offset;

	if (!buff_addr || !xsize || !ysize || index > (NUM_HW_BUFS-1) ||
		( type != CIU_HALF && type != CIU_FULL)) {
		return -1;
	}
	
	chroma = buff_addr + DMW96CIU_GET_CHROMA_OFFSET(xsize , ysize);
	luma = buff_addr;
	
	y_add_l  = ciu_dma_address[index][CIU_Y_L][type];
	y_add_h  = ciu_dma_address[index][CIU_Y_H][type];
	uv_add_l = ciu_dma_address[index][CIU_UV_L][type];
	uv_add_h = ciu_dma_address[index][CIU_UV_H][type];

	IOW32_CIU(y_add_l,  luma & 0xFFFF);	// Luma RAM - LSB
	IOW32_CIU(y_add_h,  luma >>16);		// Luma RAM - MSB
    IOW32_CIU(uv_add_l, chroma & 0xFFFF);	// Chroma RAM - LSB
	IOW32_CIU(uv_add_h, chroma >>16);		// Chroma RAM - MSB
	
	header_offset = DMW96CIU_GET_HEADER_OFFSET(xsize ,ysize ,ciu_obj.pix.pixelformat);
	header = luma + header_offset;
	header_p = (unsigned int *)(buff_virt_addr + header_offset);

	/*PIP setting*/		
	luma = header + CIU_BUFFER_HEADER;
	chroma = luma + DMW96CIU_GET_CHROMA_OFFSET(ciu_obj.pip_crect.width , ciu_obj.pip_crect.height);
	*header_p = ciu_obj.pip_crect.width;
	*(header_p+1) = ciu_obj.pip_crect.height;
	
	y_add_l  = ciu_dma_address[index][CIU_Y_L][CIU_QUARTER];
	y_add_h  = ciu_dma_address[index][CIU_Y_H][CIU_QUARTER];
	uv_add_l = ciu_dma_address[index][CIU_UV_L][CIU_QUARTER];
	uv_add_h = ciu_dma_address[index][CIU_UV_H][CIU_QUARTER];

	IOW32_CIU(y_add_l,  luma & 0xFFFF);	// Luma RAM - LSB
	IOW32_CIU(y_add_h,  luma >>16);		// Luma RAM - MSB
	IOW32_CIU(uv_add_l, chroma & 0xFFFF);	// Chroma RAM - LSB
	IOW32_CIU(uv_add_h, chroma >>16);		// Chroma RAM - MSB

	return 0;
}


/*****************************************************************

*****************************************************************/

void dmw96ciu_enb_set(const unsigned int enb)
{
	unsigned int tmp = 0;
	if (enb) {
		ciu_obj.ciu_hw_index  = 0;
	}
	tmp  = IOR32_CIU(CIU_EN) & ~(0x1 << 0);
	PDEBUG("enb = %d\n",enb);

	IOW32_CIU(CIU_EN , tmp | (enb > 0 ? 1 : 0) );

	if (ciu_interface_type == CIU_INTERFACE_MIPI) {
		msleep(100);

		if (enb) {
			IOW32_CIU(MIPI_EN_CTRL, (IOR32_CIU(MIPI_EN_CTRL) | 1));
		}
		else {
			IOW32_CIU(MIPI_EN_CTRL, (IOR32_CIU(MIPI_EN_CTRL) & ~1));
		}
	}
}

/*****************************************************************

*****************************************************************/
void dmw96ciu_enb_get(unsigned int *enb)
{
	*enb = (IOR32_CIU(CIU_EN) & ( 0x1 << 0 )) > 0 ? 1 : 0;
}

/*****************************************************************

*****************************************************************/
void dmw96ciu_oneshot_set(const unsigned int enb)
{
	unsigned int tmp = 0;

	tmp  = IOR32_CIU(CIU_EN) & ~(0x1 << 1);
	IOW32_CIU(CIU_EN , tmp | (enb > 0 ? (0x1 << 1) : 0) );
	mdelay(10);
}

/*****************************************************************

*****************************************************************/
void dmw96ciu_Nth_oneshot_set(const unsigned int enb)
{
	unsigned int tmp = 0;

	tmp  = IOR32_CIU(CIU_EN) & ~(0x1 << 2);
	IOW32_CIU(CIU_EN , tmp | (enb > 0 ? (0x1 << 2) : 0) );
}



void ciu_buf_init(void)
{
	struct ciu_bufs *bufs = &ciu_obj.bufs;
	struct ciu_bufs *hw_bufs = &ciu_obj.hw_bufs;
	int i;
	
	bufs->queue = 0;
	bufs->done = 0;
	for (i = 0; i < (NUM_BUFS - 1); i++) {
		bufs->buf[i].complete = NULL;
		bufs->buf[i].vb = NULL;
		bufs->buf[i].priv = NULL;
	}

	hw_bufs->queue = 0;
	hw_bufs->done = 0;
	for (i = 0; i < (NUM_HW_BUFS); i++) {
		ciu_switch_addreses(i , ciu_obj.dummy_buf, (unsigned int)ciu_obj.dummy_buf_virt, ciu_obj.pix.width, ciu_obj.pix.height,CIU_FULL);
		hw_bufs->buf[i].complete = NULL;
		hw_bufs->buf[i].vb = NULL;
		hw_bufs->buf[i].priv = NULL;
	}
}


int ciu_free_dummy_buf(void)
{
	int rval = 0;

	if (ciu_obj.dummy_buf_virt != NULL && ciu_obj.dummy_buf_size != 0 && ciu_obj.dummy_buf != (dma_addr_t)NULL){

		dma_free_coherent(ciu_dev, ciu_obj.dummy_buf_size, ciu_obj.dummy_buf_virt, ciu_obj.dummy_buf);
	
		ciu_obj.dummy_buf_virt = NULL;
		ciu_obj.dummy_buf_size = 0;
		ciu_obj.dummy_buf = (dma_addr_t)NULL;
	}

	return rval;
}


int ciu_alloc_dummy_buf( unsigned int size)
{
	int rval = 0;

	if (size == 0)
		return -1;

	if (ciu_obj.dummy_buf_virt != NULL || ciu_obj.dummy_buf_size != 0 || ciu_obj.dummy_buf != (dma_addr_t)NULL) {
		PDEBUG("dummy buffer already allocated, hence free and allocate again\n");
		ciu_free_dummy_buf();
	}
	ciu_obj.dummy_buf_size = PAGE_ALIGN(size);
	PDEBUG("ciu_obj.dummy_buf_size = %d\n",ciu_obj.dummy_buf_size);

	ciu_obj.dummy_buf_virt = dma_alloc_coherent(ciu_dev, ciu_obj.dummy_buf_size ,&ciu_obj.dummy_buf, GFP_KERNEL);
	#ifdef CONFIG_MACH_VERSATILE_BROADTILE
		ciu_obj.dummy_buf = ciu_obj.dummy_buf - 0x80000000;
	#endif
	if (!ciu_obj.dummy_buf_virt) {
		PDEBUG("dma_alloc_coherent size %d failed\n",ciu_obj.dummy_buf_size);
		rval = -1;
	} 
	else {
		memset (ciu_obj.dummy_buf_virt,0,ciu_obj.dummy_buf_size);
		PDEBUG("ciu dummy buff = %p\n",ciu_obj.dummy_buf_virt);
	}

	return rval;
}


int ciu_enum_framesize(struct v4l2_frmsizeenum *frms)
{
    unsigned int ifmt = 0;

	for (ifmt = 0; ifmt < NUM_CIU_CAPTURE_FORMATS; ifmt++) {
               if (frms->pixel_format == ciu_formats[ifmt].pixelformat)
                       break;
    }

	if (ifmt == NUM_CIU_CAPTURE_FORMATS)
		return -EINVAL;

	frms->type = V4L2_FRMSIZE_TYPE_STEPWISE;
	frms->stepwise.min_width = CIU_SZ_MIN_W_OUTPUT;	/* Minimum frame width [pixel] */
	frms->stepwise.max_width = CIU_SZ_MAX_W_OUTPUT;	/* Maximum frame width [pixel] */
	frms->stepwise.step_width = CIU_SZ_STEP;	/* Frame width step size [pixel] */
	frms->stepwise.min_height = CIU_SZ_MIN_H_OUTPUT;	/* Minimum frame height [pixel] */
	frms->stepwise.max_height = CIU_SZ_MAN_H_OUTPUT;	/* Maximum frame height [pixel] */
	frms->stepwise.step_height = CIU_SZ_STEP;	/* Frame height step size [pixel] */

	return 0;
}

/**
 * ciu_enum_fmt_cap - Gets more information of chosen format index and type
 * @f: Pointer to structure containing index and type of format to read from.
 *
 * Returns 0 if successful, or -EINVAL if format index or format type is
 * invalid.
 **/
int ciu_enum_fmt_cap(struct v4l2_fmtdesc *f)
{
	int index = f->index;
	enum v4l2_buf_type type = f->type;
	int rval = -EINVAL;

	if (index >= NUM_CIU_CAPTURE_FORMATS)
		goto err;

	memset(f, 0, sizeof(*f));
	f->index = index;
	f->type = type;

	switch (f->type) {
	case V4L2_BUF_TYPE_VIDEO_CAPTURE:
		rval = 0;
		break;
	default:
		goto err;
	}

	f->flags = ciu_formats[index].flags;
	strncpy(f->description, ciu_formats[index].description,
		sizeof(f->description));
	f->pixelformat = ciu_formats[index].pixelformat;
err:
	return rval;
}




/**
 * ciu_check_size - Tries size configuration for I/O images of 
 * submodule 
 * @pix_input: Pointer to V4L2 pixel format structure for input image.
 * @pix_output: Pointer to V4L2 pixel format structure for output image.
 *
 * Returns 0 if successful, or return value of configuration if 
 * there is an error. 
 **/
static int ciu_check_size(struct v4l2_pix_format *pix_output)
{
	int rval = 0;
	
	if (pix_output->width < CIU_SZ_MIN_W_OUTPUT || pix_output->height < CIU_SZ_MIN_H_OUTPUT)
		return -EINVAL;

	if (pix_output->width > CIU_SZ_MAX_W_OUTPUT || pix_output->height > CIU_SZ_MAX_W_OUTPUT)
		return -EINVAL;

	return rval;
}

/**
 * ciu_save_param - Validates input/output format parameters.
 * @pix_input: Pointer to V4L2 pixel format structure for input image.
 * @pix_output: Pointer to V4L2 pixel format structure for output image.
 *
 * Always returns 0.
 **/
int ciu_save_param(struct v4l2_pix_format *pix, unsigned int scaler_mode, struct v4l2_rect* crect)
{
	unsigned int factor = 1;

	if (!pix || !crect)
		return  -EINVAL;

	if (scaler_mode != CIU_FULL && scaler_mode != CIU_HALF)
		return -EINVAL;

	if (scaler_mode == CIU_FULL)
		factor = 1;
	if (scaler_mode ==  CIU_HALF) 
		factor = 2;

	ciu_obj.pix.width = pix->width / factor;
	ciu_obj.pix.height = pix->height / factor;
	ciu_obj.pix.pixelformat = pix->pixelformat;
	ciu_obj.pix.sizeimage = pix->sizeimage;

	ciu_obj.ciu_scaler_mode = scaler_mode;
	ciu_obj.pip_crect = *crect;

	PDEBUG("ciu_obj.pix.width=%d ciu_obj.pix.height=%d\n",ciu_obj.pix.width, ciu_obj.pix.height);
    return 0;
}

void ciu_s_fmt(struct v4l2_pix_format *pix_input)
{
	unsigned int tmp = IOR32_CIU(CIU_INIT_CTRL);	
	
	if (pix_input->pixelformat == V4L2_PIX_FMT_NV12 || pix_input->pixelformat == V4L2_PIX_FMT_NV21){
		IOW32_CIU(CIU_INIT_CTRL, tmp | (unsigned int)(1<<2) );	/* YUV- 420*/
		PDEBUG("Set CIU format to 420 CIU_INIT_CTRL = 0x%x\n", IOR32_CIU(CIU_INIT_CTRL));
	}
	else if (pix_input->pixelformat == V4L2_PIX_FMT_NV16 || pix_input->pixelformat == V4L2_PIX_FMT_NV61) {
		IOW32_CIU(CIU_INIT_CTRL, tmp & (unsigned int)(~(1<<2)) ); /* YUV- 422 */
		PDEBUG("Set CIU format to 422 CIU_INIT_CTRL = 0x%x\n", IOR32_CIU(CIU_INIT_CTRL));
	}
	
}

int ciu_s_size(struct v4l2_pix_format *pix,unsigned int scaler_mode, struct v4l2_rect* crect)
{
	if (!pix || !crect)
		return -EINVAL;

	if (scaler_mode != CIU_FULL && scaler_mode != CIU_HALF)
		return -EINVAL;

	IOW32_CIU(CIU_HSIZE,pix->width);
	IOW32_CIU(CIU_VSIZE,pix->height);	
	/*PIP settings*/
	IOW32_CIU(CIU_SHSIZE, (crect->width) );
	IOW32_CIU(CIU_SVSIZE, (crect->height / 2) );

	return 0;
}

void ciu_s_scaler(unsigned int scaler_mode)
{
	unsigned int tmp = 0;

	if ( (scaler_mode != CIU_FULL) && (scaler_mode != CIU_HALF) ) {
		PDEBUG("No 1/2 or 1/4 scaler configuration\n");
		return;
	}

	tmp = IOR32_CIU(DMA_CNTL);
	IOW32_CIU(DMA_CNTL, tmp & (unsigned int)~(1<<0) ); /* Dissable  DMA of oring image */

	if (scaler_mode == CIU_HALF) {
		tmp = IOR32_CIU(CIU_INIT_CTRL);	
		IOW32_CIU(CIU_INIT_CTRL, tmp | (unsigned int)(1<<14) ); /* enable 1/2 scale */
		tmp = IOR32_CIU(DMA_CNTL);
		IOW32_CIU(DMA_CNTL, tmp | (unsigned int)(3<<1) ); /* SCALE_YDMA_EN=1 ; SCALE _UVDMA_EN=1 */
		PDEBUG("%s(%d) CIU scaler 1/2 is used\n",__FILE__,__LINE__);
	}

	tmp = IOR32_CIU(DMA_CNTL);
	IOW32_CIU(DMA_CNTL, tmp | (unsigned int)(3<<3) ); /* PIP_YDMA_EN=1 ; PIP_UVDMA_EN=1 */
	PDEBUG("Setting PIP DMA_CNTL = 0x%x\n",tmp | (unsigned int)(3<<3) );
	tmp = IOR32_CIU(CIU_CTLAL);
	IOW32_CIU(CIU_CTLAL, tmp | (unsigned int)(1<<6) ); /* enable 1/4 scale */
	PDEBUG("Setting PIP CIU_CTLAL = 0x%x\n",tmp | (unsigned int)(1<<6) );
	PDEBUG("%s(%d) CIU scaler 1/4 is used\n",__FILE__,__LINE__);

	tmp = IOR32_CIU(DMA_CNTL);
	IOW32_CIU(DMA_CNTL, tmp | (unsigned int)(1<<0) ); /* Enable DMA of oring image */

	return;
}

int ciu_s_location(struct v4l2_rect* croprect,unsigned int scaler_mode)
{
	if (!croprect)
		return -EINVAL;

	if (scaler_mode != CIU_FULL && scaler_mode != CIU_HALF)
		return -EINVAL;

	if (scaler_mode == CIU_HALF || scaler_mode == CIU_FULL) {
		IOW32_CIU(CIU_HWINDOW , croprect->left & (unsigned int)(~0x1));
		IOW32_CIU(CIU_VWINDOW , croprect->top & (unsigned int)(~0x1));  
	}
	else
		PDEBUG("Invalid scaler settings\n");

	/*Since there is a bug when setting PIP we always take the most top left image*/
	IOW32_CIU(CIU_SWINDOW, 0 );

	return 0;
}


/**
 * ciu_s_fmt_cap - Sets I/O formats and crop and configures pipeline in CIU
 * @f: Pointer to V4L2 format structure to be filled with current output format
 *
 **/

int ciu_s_fmt_cap(struct v4l2_pix_format *pix ,unsigned int scaler_mode, struct v4l2_rect* crect)
{

	int rval = 0;
	CIU_START();
	
	if (!ciu_obj.ref_count || !pix || !crect)
		return -EINVAL;
	if (scaler_mode != CIU_FULL && scaler_mode != CIU_HALF)
		return -EINVAL;
	
	rval = ciu_check_size(pix);
	if (rval)
		goto out;
	
    rval = ciu_save_param(pix,scaler_mode, crect);
	if (rval)
		goto out;
	
	/* Allocation dummy to hold both full and PIP Image */
    rval = ciu_alloc_dummy_buf( pix->sizeimage + (pix->sizeimage / 16) + CIU_BUFFER_HEADER);
	if (rval)
		goto out;
	
	/*After we success finding format and size we have to set the CIU HW*/
	ciu_s_fmt(pix);
	ciu_s_size(pix,scaler_mode, crect);
	ciu_s_scaler(scaler_mode);
	ciu_s_location(crect,scaler_mode);
	
	CIU_END();
out:
	return rval;
}


/**
 * ciu_g_cropcap - return the default cropcap
 *
 * Returns 0 
 **/
int ciu_g_cropcap(struct v4l2_cropcap *a)
{
		/*
		a->bounds.left =  ;
		a->bounds.top = ;
		a->bounds.width = ;
        a->bounds.height = ;
		a->defrect = a->bounds;
		a->pixelaspect.numerator = 1;
		a->pixelaspect.denominator = 1;
		*/
		return 0;

}

int ciu_g_crop(struct v4l2_crop *a)
{
	if (!a)
		return -1;

	//a->c = ;

	return 0;
}


/**
 * ciu_s_crop - Sets crop rectangle size and position and queues crop operation
 * @a: Pointer to V4L2 crop structure with desired parameters.
 * @pix: Pointer to V4L2 pixel format structure with desired parameters.
 *
 * Returns 0 if successful, or -EINVAL if crop parameters are out of bounds.
 **/
int ciu_s_crop(struct v4l2_crop *a, struct v4l2_pix_format *pix)
{
	struct v4l2_crop *crop = a;
	int rval = 0;

	/*TODO: As of now it is not yet supported complete*/
	return -EINVAL;

	if (!ciu_obj.ref_count)
		return -EINVAL;

	if (crop->c.left < 0)
		crop->c.left = 0;
	if (crop->c.width < 0)
		crop->c.width = 0;
	if (crop->c.top < 0)
		crop->c.top = 0;
	if (crop->c.height < 0)
		crop->c.height = 0;

	if (crop->c.left >= pix->width)
		crop->c.left = pix->width - 1;
	if (crop->c.top >= pix->height)
		crop->c.top = pix->height - 1;

	if (crop->c.left + crop->c.width > pix->width)
		crop->c.width = pix->width - crop->c.left;
	if (crop->c.top + crop->c.height > pix->height)
		crop->c.height = pix->height - crop->c.top;


	/*CUI HW limitation:
	  The position should be mulitpule of 2
	  The size of the image sould be multiple of 2*/
	crop->c.left = crop->c.left & (unsigned int)(~0x1);
	crop->c.top = crop->c.top & (unsigned int)(~0x1);
	crop->c.width = crop->c.width & (unsigned int)(~0x3);
	crop->c.height = crop->c.height & (unsigned int)(~0x3);

	
	return rval;
}

#define BLK_DMA_LEN 0x10
void set_ciu_to_default(void)
{
	IOW32_CIU(CIU_INIT_CTRL,0x3218);	/* YUV- 422 //also enable status interrupt*/
										/* CbYCrY, CCIR656, progressive scan*/
	IOW32_CIU(CIU_CTLAL,0x10);			/* CbYCrY, CCIR601, progressive scan*/
    IOW32_CIU(CIU_CTLAH,0x8);			/* VSsync polarity High, HSync polarity High, Enable YUV-saturation */
//	IOW32_CIU(CIU_CTLBL, ((0x1 << 5) | (0x1 << 2) )); /*Make pixel position interrupt and Output FIFO full Interrupt*/

	IOW32_CIU(CIU_CTLBL, ((1 << 7) | (0x1 << 6) | (0x1 << 2) )); /*Make MIPI error, Transfer done and Output FIFO full Interrupt*/

	IOW32_CIU(CIU_INT2L,0x1); 			/* Horizontal position for interrupt issue */
	IOW32_CIU(CIU_INT2H,0x1);			/* Vertical position for interrupt issue */
	IOW32_CIU(CIU_CTLBH,0x1F);	
	IOW32_CIU(CIU_HSIZE,2048);	 	
	IOW32_CIU(CIU_VSIZE,1536);		
#if NUM_HW_BUFS == 1
	IOW32_CIU(DMA_CNTL,0x1021); /*1 buffer*/
#elif NUM_HW_BUFS == 2
	IOW32_CIU(DMA_CNTL,0x1041); /*2 buffer*/
#elif NUM_HW_BUFS == 3
	IOW32_CIU(DMA_CNTL,0x1061); /*3 buffers*/
#else
 	IOW32_CIU(DMA_CNTL,0x1081); /*4 buffers*/
#endif

	IOW32_CIU(DFLOW_CTRL,0x124);		

	IOW32_CIU(YDMA_BLK_LEN,BLK_DMA_LEN);		/* DMA transfer block length (in words)*/
	IOW32_CIU(UDMA_BLK_LEN,BLK_DMA_LEN);		/* DMA transfer block length (in words)*/
	IOW32_CIU(SYDMA_BLK_LEN,BLK_DMA_LEN); 
	IOW32_CIU(SUDMA_BLK_LEN,BLK_DMA_LEN); 	
	IOW32_CIU(PYDMA_BLK_LEN,BLK_DMA_LEN);
	IOW32_CIU(PUDMA_BLK_LEN,BLK_DMA_LEN);

	/* MIPI likes these */
	IOW32_CIU(CIU_VWINDOW, 0);
	IOW32_CIU(CIU_HWINDOW, 0);
}

/**
 * ciu_disable_block - Releases the CIU resource.
 *
 * Releases the clocks also for the last release.
 **/
int ciu_disable_block(void)
{
	if (dmw96ciu == NULL)
		return -EBUSY;
    
	PDEBUG("ciu_disable_block: old %d\n", ciu_obj.ref_count);
	mutex_lock(&(ciu_obj.ciu_mutex));
	if (ciu_obj.ref_count) {
		if (--ciu_obj.ref_count == 0) {
			PDEBUG("diable clk ciu_disable_block: old %d\n", ciu_obj.ref_count);
			clk_disable(ciu_obj.ciu_clk);
			ciu_obj.pdata->ciu_reset(1);
			mdelay(1);
			ciu_obj.pdata->ciu_reset(0);
		}
	}
	mutex_unlock(&(ciu_obj.ciu_mutex));
	PDEBUG("ciu_disable_block: new %d\n", ciu_obj.ref_count);
	PDEBUG("ciu_disable_block: fifo_full_cnt = %d\n", fifo_full_cnt);
	return ciu_obj.ref_count;
}

/**
 * ciu_enable_block - Adquires the CIU resource.
 *
 * Initializes the clocks for the first acquire.
 **/
int ciu_enable_block(void)
{
    if (dmw96ciu == NULL)
		return -EBUSY;
	
	PDEBUG("ciu_enable_block: old %d\n", ciu_obj.ref_count);

	mutex_lock(&(ciu_obj.ciu_mutex));

	ciu_obj.ref_count++;

	if (ciu_obj.ref_count == 1) {
		PDEBUG("CIU enabled %d", ciu_obj.ref_count);
		clk_enable(ciu_obj.ciu_clk);
		set_ciu_to_default();

	} else {
		PDEBUG("CIU enable: already referenced %d", ciu_obj.ref_count);
	}

	mutex_unlock(&(ciu_obj.ciu_mutex));

	return ciu_obj.ref_count;
}

/***************************
 * MIPI support
 ***************************/

void __iomem * syscfg;

#define GET_SYSCFG_ADDR( reg_offset ) ((long) syscfg + reg_offset)

static void syscfg_clear_bits( int reg, int bits )
{
	writel( (readl( GET_SYSCFG_ADDR( reg ) ) & ~bits),
			GET_SYSCFG_ADDR( reg ) );
}

static void syscfg_set_bits( int reg, int bits )
{
	writel( (readl( GET_SYSCFG_ADDR( reg ) ) | bits),
			GET_SYSCFG_ADDR( reg ) );
}

/* select parallel (not MIPI) interface */
void ciu_interface_parallel(void)
{
	PDEBUG("configuring PARALLEL\n");

	ciu_interface_type = CIU_INTERFACE_PARALLEL;

	syscfg = ioremap_nocache(DMW_SYSCFG_BASE, 0x1000);

	if (!syscfg)
		BUG();

	IOW32_CIU(MIPI_EN_CTRL, (IOR32_CIU(MIPI_EN_CTRL) & ~1));

	/* unset MIPI mode */
	syscfg_clear_bits(DMW_SYSCFG_GCR1, (1 << 8));

	iounmap(syscfg);
}

/* select MIPI interface */
void ciu_interface_mipi(int lanes)
{
	PDEBUG("configuring MIPI, %d lanes\n", lanes);

	/* set MIPI mode */
	ciu_interface_type = CIU_INTERFACE_MIPI;

	syscfg = ioremap_nocache(DMW_SYSCFG_BASE, 0x1000);

	if (!syscfg)
		BUG();

	/* set MIPI mode */
	syscfg_set_bits(DMW_SYSCFG_GCR1, (1 << 8));

	IOW32_CIU(MIPI_EN_CTRL, (IOR32_CIU(MIPI_EN_CTRL) | 1));

	IOW32_CIU(MIPI_CONTROL, 0x1E00);
	IOW32_CIU(MIPI_VREFCRL, 0x20); //recommended value in spec

	/* select lanes */
	if (lanes == 2) {
		IOW32_CIU(MIPI_CFG,0x4d);
	}
	else {
		IOW32_CIU(MIPI_CFG,0x45);
	}

///////////////////////////
	//delay the searching for sync sequence
//	if ( (image_resolution == CIF_RESOLUTION) || (image_resolution == VGA_RESOLUTION) ) {
//		IOW32_CIU(MIPI_TIMER_CFG,0x0F003400);
//	}
//	else
//	if (image_resolution == QSXGA_RESOLUTION) {
		IOW32_CIU(MIPI_TIMER_CFG,0x0F001200);
//	}
////////////////////

	/* mask all mipi interrupts */
	IOW32_CIU(MIPI_INT_MASK, 0xffffffff);

	//add current to HS amplifiers
	syscfg_clear_bits(DMW_SYSCFG_MIPI_CTL2, (0xf<<0)|(0xf<<8)|(0xf<<4));
	syscfg_set_bits(DMW_SYSCFG_MIPI_CTL2, (0xa<<0)|(0xa<<8)|(0xa<<4));

	//add skew to clock
	syscfg_clear_bits( DMW_SYSCFG_MIPI_CTL1, 0xffffffff );
//	syscfg_set_bits( DMW_SYSCFG_MIPI_CTL1, 0xB00 ); // B works for slow and 8 for typical.
	syscfg_set_bits( DMW_SYSCFG_MIPI_CTL1, 0x800 ); // B works for slow and 8 for typical.


	msleep(400);
	/* reset module */
	IOW32_CIU(MIPI_EN_CTRL, (IOR32_CIU(MIPI_EN_CTRL) & ~1));
	msleep(400);


#ifdef CONFIG_DMW96_CIU_DEBUG
	int i;

	for (i = 0; i <= 0x58; i += 4) {
		printk( KERN_ERR "CIU reg 0x%lx: 0x%lx\n", i, IOR32_CIU( i ) );
	}

	for (i = 0; i <= 0x30; i += 4) {
		printk( KERN_ERR "MIPI reg 0x%lx: 0x%lx\n", i, IOR32_CIU( 0x800 + i) );
	}
#endif

	iounmap(syscfg);
}

int ciu_clock_rate(unsigned long xclkfreq) {

	unsigned long clk_rate = 0;
	int res = 0;

	if (!dmw96ciu)
		return -EBUSY;

	if (!xclkfreq)
		return -EINVAL;

	mutex_lock(&(ciu_obj.ciu_mutex));

	clk_rate = clk_round_rate(ciu_obj.ciu_clk, xclkfreq);

	PDEBUG("clk_rate=%ld xclkfreq=%ld\n",clk_rate, xclkfreq);

	res = clk_set_rate(ciu_obj.ciu_clk, clk_rate);

	mutex_unlock(&(ciu_obj.ciu_mutex));

	return res;

}

int init_ciu(void)
{
	mutex_init(&(ciu_obj.ciu_mutex));
	spin_lock_init(&ciu_obj.lock);
	spin_lock_init(&ciu_obj.bufs.lock);

	return 0;
}



/**
 * ciu_configure_intreface - Configures CIU Control I/F related 
 * parameters. 
 **/
int ciu_configure_intreface(void)
{
	ciu_buf_init();
	return 0;
}


static int ciu_buf_process(void)
{
	struct ciu_buf *buf = NULL;
	struct ciu_bufs *hw_bufs = &ciu_obj.hw_bufs;
	struct ciu_bufs *bufs = &ciu_obj.bufs;
    unsigned int hw_done_index = 0;
	unsigned long flags;
	volatile unsigned int vacant_hw_index = 0;
	unsigned int count;
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	unsigned int offset = 0x80000000;
#else
	unsigned int offset = 0;
#endif


	spin_lock_irqsave(&bufs->lock, flags);

	/* TODO have to understand for the HW why when using 1/2 mode currimgindex is not valid instead currentpipindex is working.
	   anyway the work aroud for that problem is to use software counter
	 
	hw_done_index = ( ( IOR32_CIU(CIU_STATA) & 0x700 ) >> 8) ;

	if (hw_done_index != 0 )
		hw_done_index--;
	else
		hw_done_index = NUM_HW_BUFS - 1;
	*/
	hw_done_index = ciu_obj.ciu_hw_index;

	ciu_obj.ciu_hw_index = (ciu_obj.ciu_hw_index + 1)%(NUM_HW_BUFS);
	
	PDEBUG("hw_done_index = %d\n",hw_done_index);

	if ( hw_bufs->buf[hw_done_index].complete )
	{
        buf = &hw_bufs->buf[hw_done_index];
		buf->vb->state = VIDEOBUF_DONE;
        buf->complete(buf->vb, buf->priv);
        hw_bufs->buf[hw_done_index].complete = NULL;
		ciu_switch_addreses(hw_done_index , ciu_obj.dummy_buf, (unsigned int)ciu_obj.dummy_buf_virt , ciu_obj.pix.width, ciu_obj.pix.height,ciu_obj.ciu_scaler_mode);
	} else
	{
		PDEBUG("Dummy DMW96 CIU capture\n");
	}

	/* Find the first vacant slot at the hw bufs queue starting 2 indexes after hw_done_index
	until we pass the whole queue*/
	count = NUM_HW_BUFS - 1;
	vacant_hw_index = (hw_done_index + 2) % NUM_HW_BUFS;
	while ( hw_bufs->buf[vacant_hw_index].complete && count--){
        vacant_hw_index = (vacant_hw_index + 1) % NUM_HW_BUFS;
	}

	PDEBUG("vacant_hw_index = %d\n",vacant_hw_index);

	PDEBUG("SW queue size=%d\n", CIU_BUFS_QUEUED(bufs) );
    
    /* Dequeue from SW_Q and enqueue to SW_HW_Q */
	count = NUM_HW_BUFS - 1;
	while( ! CIU_BUFS_IS_EMPTY(bufs)  &&  ! hw_bufs->buf[vacant_hw_index].complete && count--)
	{
        buf = CIU_BUF_DONE(bufs);
		/* Copy the phy address of the buffer form the SW_Q to the HW slot */
		ciu_switch_addreses(vacant_hw_index , videobuf_to_dma_contig(buf->vb) - offset  , (unsigned int)videobuf_to_virt_contig(buf->vb) , buf->vb->width , buf->vb->height, ciu_obj.ciu_scaler_mode);
		/* Copy the call back and the is_dummy from SW_Q to the SW_HW_q */
		hw_bufs->buf[vacant_hw_index] = *buf;
		vacant_hw_index = (vacant_hw_index + 1) % NUM_HW_BUFS;
        CIU_BUF_MARK_DONE(bufs);

	}

	#ifdef CONFIG_DMW96_CIU_DEBUG
	{
		unsigned int hw_queue_size = 0;
		unsigned int i = 0;
		for (i = 0 ; i < 4 ; i++)
			if (hw_bufs->buf[i].complete){
					PDEBUG("arr[%d]=%d\n", i,hw_bufs->buf[i].vb->i);
					hw_queue_size++;
			} else
				PDEBUG("arr[%d]=%s\n", i, "Dummy");
		PDEBUG("HW queue size=%d\n", hw_queue_size );
	}
	#endif 

	spin_unlock_irqrestore(&bufs->lock, flags);

	return 0;
}

int ciu_buf_queue(struct videobuf_buffer *vb,
		  void (*complete)(struct videobuf_buffer *vb, void *priv),
		  void *priv)
{
	unsigned long flags;
	struct ciu_buf *buf;

	struct ciu_bufs *bufs = &ciu_obj.bufs;

	spin_lock_irqsave(&bufs->lock, flags);

	//BUG_ON(CIU_BUFS_IS_FULL(bufs));
	if ( CIU_BUFS_IS_FULL(bufs) ){
		printk("DMW96 CIU Queue is full %s %s(%d)\n",__FILE__ ,__func__, __LINE__);
		return -1;
	}

	buf = CIU_BUF_QUEUE(bufs);

	buf->complete = complete;
	buf->vb = vb;
	buf->priv = priv;

	CIU_BUF_MARK_QUEUE(bufs);

	PDEBUG("vb->i=%d SW Queue Size=%d\n", buf->vb->i, CIU_BUFS_QUEUED(bufs));

	spin_unlock_irqrestore(&bufs->lock, flags);

	return 0;
}

void ciu_vbq_release(struct videobuf_queue *vbq, struct videobuf_buffer *vb)
{
	ciu_free_dummy_buf();
    return;
}


/**
 * dmw96_ciu_isr - Interrupt Service Routine for Camera CIU module.
 * @irq: Not used currently.
 *
 * Handles the corresponding callback if plugged in.
 *
 * Returns IRQ_HANDLED when IRQ was correctly handled, or IRQ_NONE when the
 * IRQ wasn't handled. 
 */ 

#define CIU_TRANSFER_DONE (0x1 << 6)
#define CIU_PIXEL_POS (0x1 << 5)
#define CIU_FIFO_FULL (0x1 << 2)
static irqreturn_t dmw96_ciu_isr(int irq, void *_ciu)
{
	u32 irqstatus = 0;
	unsigned long irqflags = 0;
    
    irqstatus = IOR32_CIU(CIU_STATA);
	IOW32_CIU(CIU_INTRC , -1);

	if (irqstatus & (1 << 7)) {
		printk(KERN_ERR "MIPI error int: 0x%u\n", IOR32_CIU(MIPI_INT_STAT));

		/* disable MIPI, let CIU time out */
		IOW32_CIU(MIPI_EN_CTRL, (IOR32_CIU(MIPI_EN_CTRL) & ~1));
		IOR32_CIU(MIPI_INT_RD_CLR);

		return IRQ_HANDLED;
	}

    spin_lock_irqsave(&ciu_obj.lock, irqflags);
	
    //if (irqstatus & CIU_PIXEL_POS){
	if (irqstatus & CIU_TRANSFER_DONE) {
        ciu_buf_process();		
	}

	if (irqstatus & CIU_FIFO_FULL) {
		fifo_full_cnt++;
	}

	spin_unlock_irqrestore(&ciu_obj.lock, irqflags);

	return IRQ_HANDLED;
}


static int ciu_remove(struct platform_device *pdev)
{
	struct ciu_device *ciu = platform_get_drvdata(pdev);
    
    if (!ciu)
		return 0;

	clk_put(ciu_obj.ciu_clk);
    
    free_irq(ciu->irq, &ciu_obj);

	dmw96ciu = NULL;

	kfree(ciu);

	return 0;
	
}

#ifdef CONFIG_PM

static int ciu_suspend(struct platform_device *pdev, pm_message_t state)
{
	//int reset;

	mutex_lock(&(ciu_obj.ciu_mutex));

	PDEBUG("ciu_suspend: starting\n");
	if (ciu_obj.ref_count == 0)
		goto out;

	/*TODO Dudid add code when chip ready*/
	/*ciu_disable_interrupts();*/

	/*ciu_disb_clks();*/

out:
	PDEBUG("ciu_suspend: done\n");

	mutex_unlock(&(ciu_obj.ciu_mutex));
	return 0;
}

static int ciu_resume(struct platform_device *pdev)
{
	int ret_err = 0;

	PDEBUG("ciu_resume: starting\n");

	if (dmw96ciu == NULL)
		goto out;

	if (ciu_obj.ref_count >= 0) {
		/* TODO Dudi add in real chip
		ret_err = ciu_enb_clks();
		if (ret_err)
			goto out;
        ciu_enable_interrupts(RAW_CAPTURE(&ciu_obj));
        */
	}

out:
	PDEBUG("ciu_resume: done \n");

	return ret_err;
}

#else

#define ciu_suspend	NULL
#define ciu_resume	NULL

#endif /* CONFIG_PM */


/*TODO DudiD handle errors*/
static int ciu_probe(struct platform_device *pdev)
{
	struct ciu_device *ciu;
	struct resource *mem;
#ifdef CONFIG_MACH_VERSATILE_BROADTILE
	int intr_type = IRQF_SHARED;
#else
	int intr_type = 0;
#endif
	int ret;

	ciu_dev = &pdev->dev;

	memset(&ciu_obj,0,sizeof(ciu_obj));

    ciu = kzalloc(sizeof(*ciu), GFP_KERNEL);
	if (!ciu) {
		dev_err(&pdev->dev, "could not allocate memory\n");
		return -ENOMEM;
	}

	platform_set_drvdata(pdev, ciu);

	ciu->dev = &pdev->dev;

    /* request the mem region for the camera registers */
	
	mem = platform_get_resource(pdev, IORESOURCE_MEM, 0);
	if (!mem) {
		dev_err(ciu->dev, "no mem resource?\n");
		goto err;
	}
	if (!request_mem_region(mem->start, (mem->end - mem->start) + 1,
				pdev->name)) {
		dev_err(ciu->dev,
			"cannot reserve camera register I/O region\n");
		goto err;
	}

	dmw96ciu_regs = ioremap(mem->start, mem->end - mem->start);
	if (!dmw96ciu_regs) {
		dev_err(ciu->dev, "unable to map registers\n");
		goto err;
	}
	
    PDEBUG("ioremap CIU registres %p\n", dmw96ciu_regs);

	ciu->irq = platform_get_irq(pdev, 0);
	if (ciu->irq <= 0) {
		dev_err(ciu->dev, "no irq for camera?\n");
		return -ENODEV;
	}

	if (request_irq(ciu->irq, dmw96_ciu_isr, intr_type,"DMW96 CIU ISR", &ciu_obj)) {
		dev_err(ciu->dev, "Could not install ISR %d\n", ciu->irq);
		ret = -EINVAL;
		goto err;
	}

	ciu_obj.pdata = (struct ciu_pdata *)pdev->dev.platform_data;

	ciu_obj.ciu_clk = clk_get(&pdev->dev, NULL);
	if (IS_ERR(ciu_obj.ciu_clk)) {
		PDEBUG("clk_get for ""ciu_clk failed\n");
		return PTR_ERR(ciu_obj.ciu_clk);
	}

    ret = init_ciu();
	if (ret < 0) {
		dev_err(ciu->dev, "fail to init ciu\n");
		goto err;
	}

	ciu_obj.ref_count = 0;

	mutex_init(&(ciu_obj.ciu_mutex));
	spin_lock_init(&ciu_obj.lock);
	spin_lock_init(&ciu_obj.bufs.lock);

	dmw96ciu = ciu;

	PDEBUG("\n\n\n P r o b i n g	DMW96	C I U	 s u c c e s s  CIU_CTLBH=0x%x ! ! !\n\n\n", IOR32_CIU(CIU_CTLBH));

	return 0;

err:
	dmw96ciu = NULL;
	ciu_remove(pdev);
	return -ENODEV;
}




static struct platform_driver ciu_driver = {
	.probe	 = ciu_probe,
	.remove	 = ciu_remove,
#ifdef CONFIG_PM
	.suspend = ciu_suspend,
	.resume	 = ciu_resume,
#endif
	.driver	 = {
		.name = CIU_NAME,
		.owner = THIS_MODULE,
	},
};


static int __init ciu_init(void)
{
	return platform_driver_register(&ciu_driver);
}

static void __exit ciu_cleanup(void)
{
	platform_driver_unregister(&ciu_driver);
}

MODULE_AUTHOR("Dudi Dolev <dudi.dolev@dspg.com>");
MODULE_DESCRIPTION("DMW96 CIU Video for Linux camera driver");
MODULE_LICENSE("GPL");


module_init(ciu_init);
module_exit(ciu_cleanup);
