/*
 * wm897X_mod.c  --  WM8976 / WM8978 ALSA SoC Audio Codec driver (modified).
 *
 * Copyright (C) 2011 DSPG (DSP Group)
 *
 * Based on wm8978.c:
 *  Copyright (C) 2009-2010 Guennadi Liakhovetski <g.liakhovetski@gmx.de>
 *  Copyright (C) 2007 Carlos Munoz <carlos@kenati.com>
 *  Copyright 2006-2009 Wolfson Microelectronics PLC.
 *  Based on wm8974 and wm8990 by Liam Girdwood <lrg@slimlogic.co.uk>
 *
 * Modified by Avi Miller <Avi.Miller@dspg.com>
 *
 * 16 October, 2011:
 * Adapted to kernel 2.6.39 by by Avi Miller <Avi.Miller@dspg.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 */

//#define DEBUG
//#define V_DEBUG
//#define DEBUG_WM_REG_WRITE

#include <linux/module.h>
#include <linux/moduleparam.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/i2c.h>
#include <linux/platform_device.h>
#include <linux/regulator/consumer.h>
#include <linux/slab.h>
#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/pcm_params.h>
#include <sound/soc.h>
#include <sound/initval.h>
#include <sound/tlv.h>
#include <asm/div64.h>
#include <asm/atomic.h>
#include <linux/earlysuspend.h>
#include <linux/semaphore.h>

#include "wm897X_mod.h"

/*****************************************************************************/

#ifdef V_DEBUG
	#ifndef DEBUG 
		#define DEBUG
	#endif
	#define vdbg_prt(fmt, ...) \
		pr_debug("WM asoc Cdec-drv: %s(): " fmt, __func__, ##__VA_ARGS__)
#else
	#define vdbg_prt(fmt, ...)
#endif


#ifdef DEBUG
	#define dbg_prt(fmt, ...) \
		pr_debug("WM asoc Cdec-drv: %s(): " fmt, __func__, ##__VA_ARGS__)

	#define DIR_STR(stream_dir)  ((stream_dir == 0) ? "for Playback" : "for Capture")
#else
	#define dbg_prt(fmt, ...)
	#define DIR_SRT(dir)
#endif

/*****************************************************************************/

/* Define this to:
 *	- skip unnecessary i2c-writes to registers
 *	- avoid i2c-writes of reserved and non-existing registers
 */
/* caused some issues when returning from sleep */
//#define _ALTERNATE_WRITE_FUNCT

/*****************************************************************************/

static uint g_is_codec_master = 1;

//static uint g_opencnt = 0;
static atomic_t g_opencnt = { 0 };

/* wm897X register addresses, by name:
 * Note: some registers are Undefined for this chip !
 */
#define WM897X_RESET			 0	/* 0x00 */
#define WM897X_POWER_MANAGEMENT_1	 1	/* 0x01 */
#define WM897X_POWER_MANAGEMENT_2	 2	/* 0x02 */
#define WM897X_POWER_MANAGEMENT_3	 3	/* 0x03 */
#define WM897X_AUDIO_INTERFACE		 4	/* 0x04 */
#define WM897X_COMPANDING_CONTROL	 5	/* 0x05 */
#define WM897X_CLOCKING 		 6	/* 0x06 */
#define WM897X_ADDITIONAL_CONTROL	 7	/* 0x07 */
#define WM897X_GPIO_CONTROL		 8	/* 0x08 */
#define WM897X_JACK_DETECT_CONTROL_1	 9	/* 0x09 */
#define WM897X_DAC_CONTROL		10	/* 0x0A */
#define WM897X_LEFT_DAC_DIGITAL_VOLUME	11	/* 0x0B */
#define WM897X_RIGHT_DAC_DIGITAL_VOLUME 12	/* 0x0C */
#define WM897X_JACK_DETECT_CONTROL_2	13	/* 0x0D */
#define WM897X_ADC_CONTROL		14	/* 0x0E */
#define WM897X_LEFT_ADC_DIGITAL_VOLUME	15	/* 0x0F */
#define WM897X_RIGHT_ADC_DIGITAL_VOLUME 16	/* 0x10 */
#define WM897X_REG_17_UNDEFINED 	17	/* 0x11 */
#define WM897X_EQ1			18	/* 0x12 */
#define WM897X_EQ2			19	/* 0x13 */
#define WM897X_EQ3			20	/* 0x14 */
#define WM897X_EQ4			21	/* 0x15 */
#define WM897X_EQ5			22	/* 0x16 */
#define WM897X_REG_23_UNDEFINED 	23	/* 0x17 */
#define WM897X_DAC_LIMITER_1		24	/* 0x18 */
#define WM897X_DAC_LIMITER_2		25	/* 0x19 */
#define WM897X_REG_26_UNDEFINED 	26	/* 0x1A */
#define WM897X_NOTCH_FILTER_1		27	/* 0x1B */
#define WM897X_NOTCH_FILTER_2		28	/* 0x1C */
#define WM897X_NOTCH_FILTER_3		29	/* 0x1D */
#define WM897X_NOTCH_FILTER_4		30	/* 0x1E */
#define WM897X_REG_31_UNDEFINED 	31	/* 0x1F */
#define WM897X_ALC_CONTROL_1		32	/* 0x20 */
#define WM897X_ALC_CONTROL_2		33	/* 0x21 */
#define WM897X_ALC_CONTROL_3		34	/* 0x22 */
#define WM897X_NOISE_GATE		35	/* 0x23 */
#define WM897X_PLL_N			36	/* 0x24 */
#define WM897X_PLL_K1			37	/* 0x25 */
#define WM897X_PLL_K2			38	/* 0x26 */
#define WM897X_PLL_K3			39	/* 0x27 */
#define WM897X_REG_40_UNDEFINED 	40	/* 0x28 */
#define WM897X_3D_CONTROL		41	/* 0x29 */
#define WM897X_REG_42_UNDEFINED 	42	/* 0x2A */
#define WM897X_BEEP_CONTROL		43	/* 0x2B */
#define WM897X_INPUT_CONTROL		44	/* 0x2C */
#define WM897X_LEFT_INP_PGA_CONTROL	45	/* 0x2D */
#define WM897X_RIGHT_INP_PGA_CONTROL	46	/* 0x2E */
#define WM897X_LEFT_ADC_BOOST_CONTROL	47	/* 0x2F */
#define WM897X_RIGHT_ADC_BOOST_CONTROL	48	/* 0x30 */
#define WM897X_OUTPUT_CONTROL		49	/* 0x31 */
#define WM897X_LEFT_MIXER_CONTROL	50	/* 0x32 */
#define WM897X_RIGHT_MIXER_CONTROL	51	/* 0x33 */
#define WM897X_LOUT1_CONTROL		52	/* 0x34 */
#define WM897X_ROUT1_CONTROL		53	/* 0x35 */
#define WM897X_LOUT2_CONTROL		54	/* 0x36 */
#define WM897X_ROUT2_CONTROL		55	/* 0x37 */
#define WM897X_OUT3_MIXER_CONTROL	56	/* 0x38 */
#define WM897X_OUT4_MIXER_CONTROL	57	/* 0x39 */

/* Total number of registers (including the undefined ones) */
#define WM897X_CACHEREGNUM	58

/* The following registers contain an "update" bit - bit 8.
 * This means, that one can write new volume value for both channels,
 * but the volume is actually updated only when the "update" bit is set -
 * simultaneously for both channels.
 */
static const int update_reg[] = {
	WM897X_LEFT_DAC_DIGITAL_VOLUME,
	WM897X_RIGHT_DAC_DIGITAL_VOLUME,
	WM897X_LEFT_ADC_DIGITAL_VOLUME,
	WM897X_RIGHT_ADC_DIGITAL_VOLUME,
	WM897X_LEFT_INP_PGA_CONTROL,
	WM897X_RIGHT_INP_PGA_CONTROL,
	WM897X_LOUT1_CONTROL,
	WM897X_ROUT1_CONTROL,
	WM897X_LOUT2_CONTROL,
	WM897X_ROUT2_CONTROL,
};


/* Note: Test first(!) for definition of CONFIG_SND_SOC_WM8978_MOD, so that
 *       in case Both CONFIG_SND_SOC_WM8978_MOD and CONFIG_SND_SOC_WM8976
 *	 are defined, the code will compile for WM8978.
 *
 *	 In this case, it will work OK also for WM8976, with some additional i2c
 *	 writes of registers that do not actually exist in WM8976.
*/

/* Compilation for WM8978 ! */
#if defined (CONFIG_SND_SOC_WM8978_MOD)
#pragma message "Compiling for Wolfson WM8978 - Stereo input path !" 

/* wm8978 registers values upon Reset: */
#ifndef _ALTERNATE_WRITE_FUNCT
static const u16 wm897X_reg_reset_val[WM897X_CACHEREGNUM] = {
	0x0000, 0x0000, 0x0000, 0x0000,	/* 0x00...0x03 */
	0x0050, 0x0000, 0x0140, 0x0000,	/* 0x04...0x07 */
	0x0000, 0x0000, 0x0000, 0x00ff,	/* 0x08...0x0b */
	0x00ff, 0x0000, 0x0100, 0x00ff,	/* 0x0c...0x0f */
	0x00ff, 0x0000, 0x012c, 0x002c,	/* 0x10...0x13 */
	0x002c, 0x002c, 0x002c, 0x0000,	/* 0x14...0x17 */
	0x0032, 0x0000, 0x0000, 0x0000,	/* 0x18...0x1b */
	0x0000, 0x0000, 0x0000, 0x0000,	/* 0x1c...0x1f */
	0x0038, 0x000b, 0x0032, 0x0000,	/* 0x20...0x23 */
	0x0008, 0x000c, 0x0093, 0x00e9,	/* 0x24...0x27 */
	0x0000, 0x0000, 0x0000, 0x0000,	/* 0x28...0x2b */
	0x0033, 0x0010, 0x0010, 0x0100,	/* 0x2c...0x2f */
	0x0100, 0x0002, 0x0001, 0x0001,	/* 0x30...0x33 */
	0x0039, 0x0039, 0x0039, 0x0039,	/* 0x34...0x37 */
	0x0001,	0x0001,			/* 0x38...0x3b */
 };
#else
/* value of 0xFFFF marks Reserved or Non-Existing register in wm8978 */
static const u16 wm897X_reg_reset_val[WM897X_CACHEREGNUM] = {
	0x0000, 0x0000, 0x0000, 0x0000,	/* 0 -   3 */ 
	0x0050, 0x0000, 0x0140, 0x0000, /* 4 -   7 */
	0x0000, 0x0000, 0x0000, 0x00FF, /* 8 -  11 */
	0x00FF, 0x0000, 0x0100, 0x00FF, /* 12 - 15 */
	0x00FF, 0xFFFF, 0x012C, 0x002C, /* 16 - 19 */ // 17-NonExist
	0x002C, 0x002C, 0x002C, 0xFFFF, /* 20 - 23 */ // 23-NonExist
	0x0032, 0x0000, 0xFFFF, 0x0000, /* 24 - 27 */ // 26-NonExist
	0x0000, 0x0000, 0x0000, 0xFFFF, /* 28 - 31 */ // 31-NonExist
	0x0038, 0x000B, 0x0032, 0x0000, /* 32 - 35 */
	0x0008, 0x000C, 0x0093, 0x00E9, /* 36 - 39 */
	0xFFFF, 0x0000, 0xFFFF, 0x0000, /* 40 - 43 */ // 40-Reserved, 42-NonExist
	0x0033, 0x0010, 0x0010, 0x0100, /* 44 - 47 */
	0x0100, 0x0002, 0x0001, 0x0001, /* 48 - 51 */
	0x0039, 0x0039, 0x0039, 0x0039, /* 52 - 55 */
	0x0001, 0x0001			/* 56 - 57 */
};
#endif //_ALTERNATE_WRITE_FUNCT

/* Compilation for WM8976 ! */
#elif defined(CONFIG_SND_SOC_WM8976)
#pragma message "Compiling for Wolfson WM8976 - Mono input path !!!" 

/* wm8976 registers values upon Reset: */
#ifndef _ALTERNATE_WRITE_FUNCT
static const u16 wm897X_reg_reset_val[WM897X_CACHEREGNUM] = {
	 0x0000, 0x0000, 0x0000, 0x0000, /* 0x00...0x03 */
	 0x0050, 0x0000, 0x0140, 0x0000, /* 0x04...0x07 */
	 0x0000, 0x0000, 0x0000, 0x00ff, /* 0x08...0x0b */
	 0x00ff, 0x0000, 0x0100, 0x00ff, /* 0x0c...0x0f */
	 0x00ff, 0x0000, 0x012c, 0x002c, /* 0x10...0x13 */
	 0x002c, 0x002c, 0x002c, 0x0000, /* 0x14...0x17 */
	 0x0032, 0x0000, 0x0000, 0x0000, /* 0x18...0x1b */
	 0x0000, 0x0000, 0x0000, 0x0000, /* 0x1c...0x1f */
	 0x0038, 0x000b, 0x0032, 0x0000, /* 0x20...0x23 */
	 0x0008, 0x000c, 0x0093, 0x00e9, /* 0x24...0x27 */
	 0x0000, 0x0000, 0x0000, 0x0000, /* 0x28...0x2b */
	 0x0003, 0x0010, 0x0010, 0x0100, /* 0x2c...0x2f */
	 0x0100, 0x0002, 0x0001, 0x0001, /* 0x30...0x33 */
	 0x0039, 0x0039, 0x0039, 0x0039, /* 0x34...0x37 */
	 0x0001, 0x0001,		 /* 0x38...0x3b */
 };
#else
/* value of 0xFFFF marks Reserved or Non-Existing register in WM8976 */
static const u16 wm897X_reg_reset_val[WM897X_CACHEREGNUM] = {
	0x0000, 0x0000, 0x0000, 0x0000, /* 0 -	 3 */ 
	0x0050, 0x0000, 0x0140, 0x0000, /* 4 -	 7 */
	0x0000, 0x0000, 0x0000, 0x00FF, /* 8 -	11 */
	0x00FF, 0x0000, 0x0100, 0x00FF, /* 12 - 15 */
	0xFFFF, 0xFFFF, 0x012C, 0x002C, /* 16 - 19 */ // 16-Reserved, 17-NonExist
	0x002C, 0x002C, 0x002C, 0xFFFF, /* 20 - 23 */ // 23-NonExist
	0x0032, 0x0000, 0xFFFF, 0x0000, /* 24 - 27 */ // 26-NonExist
	0x0000, 0x0000, 0x0000, 0xFFFF, /* 28 - 31 */ // 31-NonExist
	0x0038, 0x000B, 0x0032, 0x0000, /* 32 - 35 */
	0x0008, 0x000C, 0x0093, 0x00E9, /* 36 - 39 */
	0xFFFF, 0x0000, 0xFFFF, 0x0000, /* 40 - 43 */ // 40-Reserved, 42-NonExist
	0x0003, 0x0010, 0xFFFF, 0x0100, /* 44 - 47 */ // 46-Reserved
	0xFFFF, 0x0002, 0x0001, 0x0001, /* 48 - 51 */ // 48-Reserved
	0x0039, 0x0039, 0x0039, 0x0039, /* 52 - 55 */
	0x0001, 0x0001			/* 56 - 57 */
};
#endif /* _ALTERNATE_WRITE_FUNCT */

#endif /* CONFIG_SND_SOC_WM8976 */


/* PLL divisors */
struct wm897X_pll_div {
	u32 k;    /* final 24 bits value */
	u8  n;    /* final 4 bits value. Required values: 6 to 12. */
	u8  div2; /* MCLK/2 input divider: use - '1'; bypass - '0' */
};


#ifdef _ALTERNATE_WRITE_FUNCT
typedef int (*codec_write_func_t)(struct snd_soc_codec *codec,
				  unsigned int,
				  unsigned int);

static int wm897X_reg_write(struct snd_soc_codec *codec,
			    unsigned int reg,
			    unsigned int value);

static int wm897X_uncond_reg_write(struct snd_soc_codec *codec,
				   unsigned int reg,
				   unsigned int value);
#endif /* _ALTERNATE_WRITE_FUNCT */

/* codec private data */
struct wm897X_priv {
	enum snd_soc_control_type control_type;
	void *control_data;
	struct wm897X_pll_div pll_div;
	unsigned int f_pllout;
	unsigned int f_mclk;
	unsigned int f_256fs;
	unsigned int f_opclk; /* Sourced-out clock. Unused in DSPG platforms */
	int mclk_idx;
	u16 iface_ctl;
	u16 add_ctl;
	enum wm897X_sysclk_src sysclk;
	struct regulator *avdd;
	unsigned int rate;
	unsigned int format;
	struct mutex bias_mutex;
	struct snd_soc_codec *codec;
	int is_regulator_actived;
	
#ifdef CONFIG_HAS_EARLYSUSPEND
	struct early_suspend wm897X_early_suspend;
#endif

#ifdef _ALTERNATE_WRITE_FUNCT
	codec_write_func_t	 orig_codec_write;
#endif
};

static int wm897X_set_bias_level(struct snd_soc_codec *codec,
				 enum snd_soc_bias_level level);

/*****************************************************************************/

static const char *wm897X_companding[]	= {"Off", "NC", "u-law", "A-law"};
static const char *wm897X_eqmode[]	= {"Capture", "Playback"};
static const char *wm897X_bw[]		= {"Narrow", "Wide"};
static const char *wm897X_eq1[]		= {"80Hz", "105Hz", "135Hz", "175Hz"};
static const char *wm897X_eq2[]		= {"230Hz", "300Hz", "385Hz", "500Hz"};
static const char *wm897X_eq3[]		= {"650Hz", "850Hz", "1.1kHz", "1.4kHz"};
static const char *wm897X_eq4[]		= {"1.8kHz", "2.4kHz", "3.2kHz", "4.1kHz"};
static const char *wm897X_eq5[]		= {"5.3kHz", "6.9kHz", "9kHz", "11.7kHz"};
static const char *wm897X_alc3[]	= {"ALC", "Limiter"};
static const char *wm897X_alc1[]	= {"Off", "Right", "Left", "Both"};
static const char *wm897X_pga_boost[]	= {"0db", "+20db"};

static const SOC_ENUM_SINGLE_DECL(adc_compand,	WM897X_COMPANDING_CONTROL, 1,	wm897X_companding);
static const SOC_ENUM_SINGLE_DECL(dac_compand,	WM897X_COMPANDING_CONTROL, 3,	wm897X_companding);
static const SOC_ENUM_SINGLE_DECL(eqmode,	WM897X_EQ1,		   8,	wm897X_eqmode);
static const SOC_ENUM_SINGLE_DECL(eq1,		WM897X_EQ1,		   5,	wm897X_eq1);
static const SOC_ENUM_SINGLE_DECL(eq2bw,	WM897X_EQ2,		   8,	wm897X_bw);
static const SOC_ENUM_SINGLE_DECL(eq2, 		WM897X_EQ2,		   5,	wm897X_eq2);
static const SOC_ENUM_SINGLE_DECL(eq3bw,	WM897X_EQ3,		   8,	wm897X_bw);
static const SOC_ENUM_SINGLE_DECL(eq3,		WM897X_EQ3,		   5,	wm897X_eq3);
static const SOC_ENUM_SINGLE_DECL(eq4bw,	WM897X_EQ4,		   8,	wm897X_bw);
static const SOC_ENUM_SINGLE_DECL(eq4,		WM897X_EQ4,		   5,	wm897X_eq4);
static const SOC_ENUM_SINGLE_DECL(eq5,		WM897X_EQ5,		   5,	wm897X_eq5);
static const SOC_ENUM_SINGLE_DECL(alc3,		WM897X_ALC_CONTROL_3,	   8,	wm897X_alc3);
static const SOC_ENUM_SINGLE_DECL(alc1,		WM897X_ALC_CONTROL_1,	   7,	wm897X_alc1);
static const SOC_ENUM_SINGLE_DECL(l_pgaboost,	WM897X_LEFT_ADC_BOOST_CONTROL,  8, wm897X_pga_boost);
static const SOC_ENUM_SINGLE_DECL(r_pgaboost,	WM897X_RIGHT_ADC_BOOST_CONTROL, 8, wm897X_pga_boost);

static const DECLARE_TLV_DB_SCALE(digital_tlv,	-12750,	 50, 1); /* min:-127.50db step:0.50db, mute (-127.5db sets Mute) */
static const DECLARE_TLV_DB_SCALE(eq_tlv,	 -1200,	100, 0); /* min: -12.00db step:1.00db, no mute */
static const DECLARE_TLV_DB_SCALE(inpga_tlv,	 -1200,	 75, 0); /* min: -12.00db step:0.75db, no mute */
static const DECLARE_TLV_DB_SCALE(spk_tlv,	 -5700,	100, 0); /* min: -57.00db step:1.00db, no mute */
static const DECLARE_TLV_DB_SCALE(in_boost_tlv,	 -1500,	300, 1); /* min: -15.00db step:3.00db, mute */
static const DECLARE_TLV_DB_SCALE(limboost_tlv,	     0,	100, 0); /* min:   0.00db step:1.00db, no mute */
static const DECLARE_TLV_DB_SCALE(mix_tlv,	 -1500,	300, 0); /* min: -15.00db step:3.00db, no mute */
static const DECLARE_TLV_DB_SCALE(beep_tlv,	 -1500,	300, 0); /* min: -15.00db step:3.00db, no mute */

static const struct snd_kcontrol_new wm897X_snd_controls[] = {

	/* Various controls:
	 * ----------------
	 */

	/* Digital loopback (test only) */
	SOC_SINGLE("Digital-Loopback", WM897X_COMPANDING_CONTROL, 0, 1, 0),

	/* ADCs & DACs Left/Right Swap */
	SOC_SINGLE("ADCs-Swap", WM897X_AUDIO_INTERFACE, 1, 1, 0),
	SOC_SINGLE("DACs-Swap", WM897X_AUDIO_INTERFACE, 2, 1, 0),
	
	/* ADCs & DACs Companding */
	SOC_ENUM("ADC-Companding", adc_compand),
	SOC_ENUM("DAC-Companding", dac_compand),

	/* ADCs & DACs inversion */
	SOC_SINGLE("ADC-L-Inverse", WM897X_ADC_CONTROL, 0, 1, 0),
	SOC_SINGLE("ADC-R-Inverse", WM897X_ADC_CONTROL, 1, 1, 0),

	SOC_SINGLE("DAC-L-Inverse", WM897X_DAC_CONTROL, 0, 1, 0),
	SOC_SINGLE("DAC-R-Inverse", WM897X_DAC_CONTROL, 1, 1, 0),

	/* ADCs & DACs digital Volume: */
	SOC_DOUBLE_R_TLV("ADC Capture Volume",
		WM897X_LEFT_ADC_DIGITAL_VOLUME, WM897X_RIGHT_ADC_DIGITAL_VOLUME,
		0, 255, 0, digital_tlv),
	SOC_DOUBLE_R_TLV("DAC Playback Volume",
		WM897X_LEFT_DAC_DIGITAL_VOLUME, WM897X_RIGHT_DAC_DIGITAL_VOLUME,
		0, 255, 0, digital_tlv),

	/* High-Pass filter */
	SOC_SINGLE("ADC-HPass-Filter", WM897X_ADC_CONTROL, 8, 1, 0),
	SOC_SINGLE("ADC-HPass-APP",    WM897X_ADC_CONTROL, 7, 1, 0),
	SOC_SINGLE("ADC-HPass-CutOff", WM897X_ADC_CONTROL, 4, 7, 0),

	/* Equalizer */
	SOC_ENUM("Equaliser-Functions_In", eqmode),
	SOC_ENUM("EQ1-CutOff", eq1),
	SOC_SINGLE_TLV("EQ1 Volume", WM897X_EQ1, 0, 24, 1, eq_tlv),
	SOC_ENUM("EQ2-BandWidth", eq2bw),
	SOC_ENUM("EQ2-CutOff", eq2),
	SOC_SINGLE_TLV("EQ2 Volume", WM897X_EQ2, 0, 24, 1, eq_tlv),
	SOC_ENUM("EQ3-BandWidth", eq3bw),
	SOC_ENUM("EQ3-CutOff", eq3),
	SOC_SINGLE_TLV("EQ3 Volume", WM897X_EQ3, 0, 24, 1, eq_tlv),
	SOC_ENUM("EQ4-BandWidth", eq4bw),
	SOC_ENUM("EQ4-CutOff", eq4),
	SOC_SINGLE_TLV("EQ4 Volume", WM897X_EQ4, 0, 24, 1, eq_tlv),
	SOC_ENUM("EQ5-CutOff", eq5),
	SOC_SINGLE_TLV("EQ5 Volume", WM897X_EQ5, 0, 24, 1, eq_tlv),

	/* DACs limiter */
	SOC_SINGLE("DAC-Limiter_Switch",    WM897X_DAC_LIMITER_1, 8,  1, 0),
	SOC_SINGLE("DAC-Limiter_Decay",     WM897X_DAC_LIMITER_1, 4, 15, 0),
	SOC_SINGLE("DAC-Limiter_Attack",    WM897X_DAC_LIMITER_1, 0, 15, 0),
	SOC_SINGLE("DAC-Limiter_Threshold", WM897X_DAC_LIMITER_2, 4,  5, 0),
	SOC_SINGLE_TLV("DAC-Limboost Playback Volume",	WM897X_DAC_LIMITER_2, 0, 12, 0, limboost_tlv),

	/* ALC on input path */
	SOC_ENUM("ALC-Switch", alc1),
	SOC_SINGLE("ALC-Min_Gain", WM897X_ALC_CONTROL_1, 0,  7, 0),
	SOC_SINGLE("ALC-Max_Gain", WM897X_ALC_CONTROL_1, 3,  7, 0),
	SOC_SINGLE("ALC-Hold",	   WM897X_ALC_CONTROL_2, 4, 10, 0),
	SOC_SINGLE("ALC-Target",   WM897X_ALC_CONTROL_2, 0, 15, 0),
	SOC_ENUM("ALC-Mode", alc3),
	SOC_SINGLE("ALC-Decay",    WM897X_ALC_CONTROL_3, 4, 10, 0),
	SOC_SINGLE("ALC-Attack",   WM897X_ALC_CONTROL_3, 0, 10, 0),
	SOC_SINGLE("ALC-NoiseGate_Switch",    WM897X_NOISE_GATE, 3, 1, 0),
	SOC_SINGLE("ALC-NoiseGate_Threshold", WM897X_NOISE_GATE, 0, 7, 0),

	/* DAC/ADC oversampling */
	SOC_SINGLE("DAC-128x_Oversample", WM897X_DAC_CONTROL, 3, 1, 0),
	SOC_SINGLE("ADC-128x_Oversample", WM897X_ADC_CONTROL, 3, 1, 0),


	/* Outputs Analog controls:
	 * -----------------------
	 */
	
	/* L/R OUT1 - Zero-Cross switch */
	SOC_DOUBLE_R("LR_OUT1-ZC-Switch",
		WM897X_LOUT1_CONTROL, WM897X_ROUT1_CONTROL, 7, 1, 0),
	/* L/R OUT1 - Analog Volume */
	SOC_DOUBLE_R_TLV("LR_OUT1 Playback Volume",
		WM897X_LOUT1_CONTROL, WM897X_ROUT1_CONTROL,
		0, 63, 0, spk_tlv),

	/* L/R OUT2 - Zero-Cross switch */
	SOC_DOUBLE_R("LR_OUT2-ZC-Switch",
		WM897X_LOUT2_CONTROL, WM897X_ROUT2_CONTROL, 7, 1, 0),
	/* L/R OUT2 - Analog Volume */
	SOC_DOUBLE_R_TLV("LR_OUT2 Playback Volume",
		WM897X_LOUT2_CONTROL, WM897X_ROUT2_CONTROL,
		0, 63, 0, spk_tlv),

	/* L/R Mixers AUX input Volume */
	SOC_DOUBLE_R_TLV("LR_Mix-AUX Playback Volume",
		WM897X_LEFT_MIXER_CONTROL, WM897X_RIGHT_MIXER_CONTROL,
		6, 7, 0, mix_tlv),
	/* L/R Mixers Bypass input Volume */
	SOC_DOUBLE_R_TLV("LR_Mix-Bypass Playback Volume",
		WM897X_LEFT_MIXER_CONTROL, WM897X_RIGHT_MIXER_CONTROL,
		2, 7, 0, mix_tlv),

	/* BEEP Mixer (no DAPM on this one!): */
	/* - Input AUXR Analog Volume */
	SOC_SINGLE_TLV("BEEP Playback Volume", WM897X_BEEP_CONTROL, 1, 7, 0, beep_tlv),
	/* - Input: AUXR */
	SOC_SINGLE("BEEP_Mix-Beep_Input", WM897X_BEEP_CONTROL, 0, 1, 0), /* BEEPEN */
	/* - Input: Inverted Right-Mixer output */
	SOC_SINGLE("BEEP_Mix-Inv-R_Mix_Input", WM897X_BEEP_CONTROL, 5, 1, 0), /* MUTERPGA2INV */
	
	/* - ROUT2 sourced from: Either Right-Mixer outpout Or from Beep Mixer output */
	SOC_SINGLE("ROUT2_Inverse_Switch", WM897X_BEEP_CONTROL, 4, 1, 0),


	/* Input Analog controls:
	 * ---------------------
	 */
	
	/* Input PGAs - Zero-Cross switch, Analog Volume */
	SOC_DOUBLE_R("Input-PGA-ZC-Switch",
		WM897X_LEFT_INP_PGA_CONTROL, WM897X_RIGHT_INP_PGA_CONTROL,
		7, 1, 0),
	SOC_DOUBLE_R_TLV("Input-PGA Capture Volume",
		WM897X_LEFT_INP_PGA_CONTROL, WM897X_RIGHT_INP_PGA_CONTROL,
		0, 63, 0, inpga_tlv),

	/* Inputs to input-Boost-Mixers (second-stage input mixers,
	 * after input PGAs mixers)
	 */
	SOC_ENUM("Left-Input-PGA-Boost",  l_pgaboost),
	SOC_ENUM("Right-Input-PGA-Boost", r_pgaboost),

	SOC_DOUBLE_R_TLV("Input_L2-R2-Boost Capture Volume",
		WM897X_LEFT_ADC_BOOST_CONTROL, WM897X_RIGHT_ADC_BOOST_CONTROL,
		4, 7, 0, in_boost_tlv),
	SOC_DOUBLE_R_TLV("Input_AUX-Boost Capture Volume",
		WM897X_LEFT_ADC_BOOST_CONTROL, WM897X_RIGHT_ADC_BOOST_CONTROL,
		0, 7, 0, in_boost_tlv),

	/* Input digital controls:
	 * -----------------------
	 */
	SOC_DOUBLE_R("Input-PGA1 Capture Switch",
		WM897X_LEFT_INP_PGA_CONTROL, WM897X_RIGHT_INP_PGA_CONTROL, 6, 1, 1),
};


/* Analog mixers: */

/* Left Input PGA Ampilifier, as Mixer */
static const struct snd_kcontrol_new wm897X_left_input_mixer[] = {
	SOC_DAPM_SINGLE("L2_Input-Switch",  WM897X_INPUT_CONTROL, 2, 1, 0),
	SOC_DAPM_SINGLE("LIN_Input-Switch", WM897X_INPUT_CONTROL, 1, 1, 0),
	SOC_DAPM_SINGLE("LIP_Input-Switch", WM897X_INPUT_CONTROL, 0, 1, 0),
};

/* Right Input PGA Ampilifier, as Mixer */
static const struct snd_kcontrol_new wm897X_right_input_mixer[] = {
	SOC_DAPM_SINGLE("R2_Input-Switch",  WM897X_INPUT_CONTROL, 6, 1, 0),
	SOC_DAPM_SINGLE("RIN_Input-Switch", WM897X_INPUT_CONTROL, 5, 1, 0),
	SOC_DAPM_SINGLE("RIP_Input-Switch", WM897X_INPUT_CONTROL, 4, 1, 0),
};

/* Left Output Mixer */
static const struct snd_kcontrol_new wm897X_left_out_mixer[] = {
	SOC_DAPM_SINGLE("L_Mix-LDAC_Input",         WM897X_LEFT_MIXER_CONTROL, 0, 1, 0), /* DACL2LMIX */
	SOC_DAPM_SINGLE("L_Mix-RDAC_Input",         WM897X_OUTPUT_CONTROL,     5, 1, 0), /* DACR2LMIX */
	SOC_DAPM_SINGLE("L_Mix-AUXL_Input",	    WM897X_LEFT_MIXER_CONTROL, 5, 1, 0), /* AUXL2LMIX */
	SOC_DAPM_SINGLE("L_Mix-LineBypass_L_Input", WM897X_LEFT_MIXER_CONTROL, 1, 1, 0), /* BYPL2LMIX */
};

/* Right Output Mixer */
static const struct snd_kcontrol_new wm897X_right_out_mixer[] = {
	SOC_DAPM_SINGLE("R_Mix-RDAC_Input",	    WM897X_RIGHT_MIXER_CONTROL, 0, 1, 0), /* DACR2RMIX */
	SOC_DAPM_SINGLE("R_Mix-LDAC_Input",	    WM897X_OUTPUT_CONTROL,      6, 1, 0), /* DACL2RMIX */
	SOC_DAPM_SINGLE("R_Mix-AUXR_Input",	    WM897X_RIGHT_MIXER_CONTROL, 5, 1, 0), /* AUXR2RMIX */
	SOC_DAPM_SINGLE("R_Mix-LineBypass_R_Input", WM897X_RIGHT_MIXER_CONTROL, 1, 1, 0), /* BYPR2RMIX */
};

/* OUT3 ("Left") output mixer */
static const struct snd_kcontrol_new wm897X_out3_mixer[] = {
	SOC_DAPM_SINGLE("OUT3_Mix-LDAC_Input",     WM897X_OUT3_MIXER_CONTROL, 0, 1, 0), /* LDAC2OUT3 */
	SOC_DAPM_SINGLE("OUT3_Mix-LMix_Input",     WM897X_OUT3_MIXER_CONTROL, 1, 1, 0), /* LMIX2OUT3 */
	SOC_DAPM_SINGLE("OUT3_Mix-InvOUT4_Input",  WM897X_OUT3_MIXER_CONTROL, 3, 1, 0), /* OUT4_2OUT3 */
	SOC_DAPM_SINGLE("OUT3_Mix-ByPass_L_Input", WM897X_OUT3_MIXER_CONTROL, 2, 1, 0), /* BYPL2OUT3 */
};

/* OUT4 ("Right") output mixer */
static const struct snd_kcontrol_new wm897X_out4_mixer[] = {
	SOC_DAPM_SINGLE("OUT4_Mix-RDAC_Input",	   WM897X_OUT4_MIXER_CONTROL, 0, 1, 0), /* RDAC2OUT4 */ 
	SOC_DAPM_SINGLE("OUT4_Mix-RMix_Input",	   WM897X_OUT4_MIXER_CONTROL, 1, 1, 0), /* RMIX2OUT4 */
	SOC_DAPM_SINGLE("OUT4_Mix-LDAC_Input",	   WM897X_OUT4_MIXER_CONTROL, 3, 1, 0), /* LDAC2OUT4 */
	SOC_DAPM_SINGLE("OUT4_Mix-LMix_Input",	   WM897X_OUT4_MIXER_CONTROL, 4, 1, 0), /* LMIX2OUT4 */
	SOC_DAPM_SINGLE("OUT4_Mix-ByPass_R_Input", WM897X_OUT4_MIXER_CONTROL, 2, 1, 0), /* BYPR2OUT4 */
};

static const struct snd_kcontrol_new wm897X_lout1_output[] = {
	SOC_DAPM_SINGLE("Playback Switch",WM897X_LOUT1_CONTROL,6, 1, 1),
};

static const struct snd_kcontrol_new wm897X_rout1_output[] = {
	SOC_DAPM_SINGLE("Playback Switch",WM897X_ROUT1_CONTROL,6, 1, 1),
};

static const struct snd_kcontrol_new wm897X_lout2_output[] = {
	SOC_DAPM_SINGLE("Playback Switch",WM897X_LOUT2_CONTROL,6, 1, 1),
};

static const struct snd_kcontrol_new wm897X_rout2_output[] = {
	SOC_DAPM_SINGLE("Playback Switch",WM897X_ROUT2_CONTROL,6, 1, 1),
};
static const struct snd_kcontrol_new wm897X_out3_output[] = {
	SOC_DAPM_SINGLE("Playback Switch",WM897X_OUT3_MIXER_CONTROL,6, 1, 1),
};
static const struct snd_kcontrol_new wm897X_out4_output[] = {
	SOC_DAPM_SINGLE("Playback Switch",WM897X_OUT4_MIXER_CONTROL,6, 1, 1),
};

static const char *input_type[] = {
	"None",
	"Aux",
	"Line",
	"Mic",
};

static const struct soc_enum in_mux_enum = 
	SOC_ENUM_SINGLE(SND_SOC_NOPM,0,4,input_type);

static const struct snd_kcontrol_new left_in_mux = 
	SOC_DAPM_ENUM_VIRT("Left-Input Mux", in_mux_enum);

static const struct snd_kcontrol_new right_in_mux = 
	SOC_DAPM_ENUM_VIRT("Right-Input Mux", in_mux_enum);


/* DAPM (Dynamic Audio Power Management) Widgets:
 *  each Wolfson unit here gets Enabled/Disabled "automatically",
 *  based on a "Need basis", which depends on the active route
 *  selected out of the dapm route map (audio_route[] below).
 */
static const struct snd_soc_dapm_widget wm897X_dapm_widgets[] = {
	/* Left ADC */
	SND_SOC_DAPM_ADC("Left-ADC", "Left HiFi Capture",
			 WM897X_POWER_MANAGEMENT_2, 0, 0),
	/* Right ADC */
	SND_SOC_DAPM_ADC("Right-ADC", "Right HiFi Capture",
			 WM897X_POWER_MANAGEMENT_2, 1, 0),
			 
	/* Left DAC */
	SND_SOC_DAPM_DAC("Left-DAC", "Left HiFi Playback",
			 WM897X_POWER_MANAGEMENT_3, 0, 0),
	/* Right DAC */
	SND_SOC_DAPM_DAC("Right-DAC", "Right HiFi Playback",
			 WM897X_POWER_MANAGEMENT_3, 1, 0),
			 
	/* Left Input Boost-Mixer extra inputs */
	SOC_MIXER_ARRAY("Left-Input-PGA-Mixer",  WM897X_POWER_MANAGEMENT_2,
			2, 0, wm897X_left_input_mixer),
	SND_SOC_DAPM_PGA("Left-Input-Boost",  WM897X_POWER_MANAGEMENT_2,
			4, 0, NULL, 0),

	SND_SOC_DAPM_VIRT_MUX("Left-Input Mux", SND_SOC_NOPM,0,0,&left_in_mux),

	/* Right Input Boost-Mixer extra inputs */
	SOC_MIXER_ARRAY("Right-Input-PGA-Mixer", WM897X_POWER_MANAGEMENT_2,
			3, 0, wm897X_right_input_mixer),
	SND_SOC_DAPM_PGA("Right-Input-Boost", WM897X_POWER_MANAGEMENT_2,
			5, 0, NULL, 0),

	SND_SOC_DAPM_VIRT_MUX("Right-Input Mux", SND_SOC_NOPM,0,0,&right_in_mux),

	/* Left Output Mixer */
	SOC_MIXER_ARRAY("Left-Output-Mixer", WM897X_POWER_MANAGEMENT_3,
			2, 0, wm897X_left_out_mixer),
	/* Right Output Mixer */			
	SOC_MIXER_ARRAY("Right-Output-Mixer", WM897X_POWER_MANAGEMENT_3,
			3, 0, wm897X_right_out_mixer),

	/* OUT3 Output Mixer */
	SOC_MIXER_ARRAY("Out3-Mixer", WM897X_POWER_MANAGEMENT_1,
			6, 0, wm897X_out3_mixer),
	/* OUT4 Output Mixer */
	SOC_MIXER_ARRAY("Out4-Mixer", WM897X_POWER_MANAGEMENT_1,
			7, 0, wm897X_out4_mixer),

	SND_SOC_DAPM_PGA("LOUT1 Amp", WM897X_POWER_MANAGEMENT_2,
			 7, 0,NULL,0),
	SND_SOC_DAPM_PGA("ROUT1 Amp", WM897X_POWER_MANAGEMENT_2,
			 8, 0, NULL,0),
	SND_SOC_DAPM_PGA("LOUT2 Amp", WM897X_POWER_MANAGEMENT_3,
			 6, 0, NULL,0),
	SND_SOC_DAPM_PGA("ROUT2 Amp", WM897X_POWER_MANAGEMENT_3,
			 5, 0, NULL,0),

	SND_SOC_DAPM_PGA("OUT3 Amp",  WM897X_POWER_MANAGEMENT_3,
			   7, 0, NULL, 0),
	SND_SOC_DAPM_PGA("OUT4 Amp",  WM897X_POWER_MANAGEMENT_3,
			   8, 0, NULL, 0),

	/* Outputs control: */
	SND_SOC_DAPM_SWITCH("LOUT1-En", SND_SOC_NOPM,
			 0, 0,wm897X_lout1_output),
	SND_SOC_DAPM_SWITCH("ROUT1-En", SND_SOC_NOPM,
			 0, 0, wm897X_rout1_output),
	SND_SOC_DAPM_SWITCH("LOUT2-En", SND_SOC_NOPM,
			 0, 0, wm897X_lout2_output),
	SND_SOC_DAPM_SWITCH("ROUT2-En", SND_SOC_NOPM,
			 0, 0, wm897X_rout2_output),
	SND_SOC_DAPM_SWITCH("OUT3-En", SND_SOC_NOPM,
			 0, 0, wm897X_out3_output),
	SND_SOC_DAPM_SWITCH("OUT4-En", SND_SOC_NOPM,
			 0, 0, wm897X_out4_output),



	/* Mic(s) Bias */
	SND_SOC_DAPM_MICBIAS("Mic-Bias", WM897X_POWER_MANAGEMENT_1, 4, 0),

	/* Input endpoint pins: */
	SND_SOC_DAPM_INPUT("LIN"),
	SND_SOC_DAPM_INPUT("RIN"),
	SND_SOC_DAPM_INPUT("LIP"),
	SND_SOC_DAPM_INPUT("RIP"),
	SND_SOC_DAPM_INPUT("L2"),
	SND_SOC_DAPM_INPUT("R2"),
	SND_SOC_DAPM_INPUT("AUXL"),
	SND_SOC_DAPM_INPUT("AUXR"),

	/* Output endpoint pins: */
	SND_SOC_DAPM_OUTPUT("LOUT1"),
	SND_SOC_DAPM_OUTPUT("ROUT1"),
	SND_SOC_DAPM_OUTPUT("LOUT2"),
	SND_SOC_DAPM_OUTPUT("ROUT2"),
	SND_SOC_DAPM_OUTPUT("OUT3"),
	SND_SOC_DAPM_OUTPUT("OUT4"),
};

static const struct snd_soc_dapm_route audio_route[] = {
	
	/* sink, control, source (, connected) : */

	/* Output path: */
	
	/* DACs, etc into Left/Right Output mixers */
	{"Left-Output-Mixer", "L_Mix-LDAC_Input", "Left-DAC"},
	{"Left-Output-Mixer", "L_Mix-AUXL_Input", "AUXL"},
	{"Left-Output-Mixer", "L_Mix-LineBypass_L_Input", "Left-Input-Boost"},
	{"Left-Output-Mixer", "L_Mix-RDAC_Input", "Right-DAC"},

	{"Right-Output-Mixer", "R_Mix-RDAC_Input", "Right-DAC"},
	{"Right-Output-Mixer", "R_Mix-AUXR_Input", "AUXR"},
	{"Right-Output-Mixer", "R_Mix-LineBypass_R_Input", "Right-Input-Boost"},
	{"Right-Output-Mixer", "R_Mix-LDAC_Input", "Left-DAC"},

	/* DACs, etc into Out3/Out4 Output mixers */
	{"Out3-Mixer", "OUT3_Mix-LDAC_Input", "Left-DAC"},
	{"Out3-Mixer", "OUT3_Mix-InvOUT4_Input", "Out4-Mixer"},
	{"Out3-Mixer", "OUT3_Mix-LMix_Input", "Left-Output-Mixer"},
	{"Out3-Mixer", "OUT3_Mix-ByPass_L_Input", "Left-Input-Boost"},

	{"Out4-Mixer", "OUT4_Mix-RDAC_Input", "Right-DAC"},
	{"Out4-Mixer", "OUT4_Mix-LDAC_Input", "Left-DAC"},
	{"Out4-Mixer", "OUT4_Mix-RMix_Input", "Right-Output-Mixer"},
	{"Out4-Mixer", "OUT4_Mix-LMix_Input", "Left-Output-Mixer"},
	{"Out4-Mixer", "OUT4_Mix-ByPass_R_Input", "Right-Input-Boost"},

	/* Output mixers into output pins, "via" Mute switches and Volume controls */
	{"ROUT1 Amp", NULL, "Right-Output-Mixer"},
	{"ROUT1-En", "Playback Switch", "ROUT1 Amp"},
	{"ROUT1",NULL,"ROUT1-En"},

	{"LOUT1 Amp", NULL, "Left-Output-Mixer"},
	{"LOUT1-En", "Playback Switch", "LOUT1 Amp"},
	{"LOUT1",NULL,"LOUT1-En"},
		
	{"ROUT2 Amp", NULL, "Right-Output-Mixer"},
	{"ROUT2-En", "Playback Switch", "ROUT2 Amp"},
	{"ROUT2",NULL,"ROUT2-En"},

	{"LOUT2 Amp", NULL, "Left-Output-Mixer"},
	{"LOUT2-En", "Playback Switch", "LOUT2 Amp"},
	{"LOUT2",NULL,"LOUT2-En"},

	{"OUT3 Amp", NULL, "Out3-Mixer"},
	{"OUT3-En", "Playback Switch", "OUT3 Amp"},
	{"OUT3", NULL, "OUT3-En"},
	
	{"OUT4 Amp", NULL, "Out4-Mixer"},
	{"OUT4-En", "Playback Switch", "OUT4 Amp"},
	{"OUT4", NULL, "OUT4-En"},

	/* Input path: */

	/* Mic #1 & L2 input pins into Left input PGA Amp + this PGA Mute */
	{"Left-Input-PGA-Mixer", "L2_Input-Switch", "L2"},
	{"Left-Input-PGA-Mixer", "LIN_Input-Switch", "LIN"},
	{"Left-Input-PGA-Mixer", "LIP_Input-Switch", "LIP"},

	/* Mic #2 & R2 input pins into Right input PGA Amp + this PGA Mute */
	{"Right-Input-PGA-Mixer", "R2_Input-Switch", "R2"},
	{"Right-Input-PGA-Mixer", "RIN_Input-Switch", "RIN"},
	{"Right-Input-PGA-Mixer", "RIP_Input-Switch", "RIP"},

	/* Extra Left input pins + Left input PGA Amp
	 * into Left input Boost Mixer, to Left ADC
	 */
	{"Left-Input-Boost", NULL, "Left-Input-PGA-Mixer"},

	{"Left-Input Mux", "Mic", "Left-Input-Boost"},
	{"Left-Input Mux", "Line", "L2"},
	{"Left-Input Mux", "Aux", "AUXL"},
	
	{"Left-ADC", NULL, "Left-Input Mux"},

	/* Extra Right input pins + Right input PGA Amp
	 * into Right input Boost Mixer, to Right ADC
	 */
	{"Right-Input-Boost", NULL, "Right-Input-PGA-Mixer"},

	{"Right-Input Mux", "Mic", "Right-Input-Boost"},
	{"Right-Input Mux", "Aux", "AUXR"},
	{"Right-Input Mux", "Line", "R2"},

	{"Right-ADC", NULL, "Right-Input Mux"},
};

static int wm897X_add_widgets(struct snd_soc_codec *codec)
{
	struct snd_soc_dapm_context *dapm = &codec->dapm;

	dbg_prt();

	/* Add wm897X DAPM widgets */
	snd_soc_dapm_new_controls(dapm, wm897X_dapm_widgets,
				  ARRAY_SIZE(wm897X_dapm_widgets));
	
	/* Set up the wm897X audio-routes map */
	snd_soc_dapm_add_routes(dapm, audio_route, ARRAY_SIZE(audio_route));

	return 0;
}


#define FIXED_PLL_SIZE (1 << 24)

static void wm897X_pll_factors(struct snd_soc_codec *codec,
			struct wm897X_pll_div *pll_div,
			unsigned int target,
			unsigned int source)
{
	u64 k_part;
	unsigned int k, n_div, n_mod;

	dbg_prt();

	n_div = target / source;
	if (n_div < 6) {
		source >>= 1;
		pll_div->div2 = 1;
		n_div = target / source;
	} else {
		pll_div->div2 = 0;
	}

	if (n_div < 6 || n_div > 12) {
		dev_warn(codec->dev,
			 "wm897X N value exceeds recommended range! N = %u\n",
		 	 n_div);
	}

	pll_div->n = n_div;
	n_mod = target - source * n_div;
	k_part = FIXED_PLL_SIZE * (long long)n_mod + source / 2;

	do_div(k_part, source);

	k = k_part & 0xFFFFFFFF;

	pll_div->k = k;
}

/* MCLK dividers:
 * MCLKDIV divides by 1, 1.5, 2, 3, 4, 6, 8, 12
 */
static const int mclk_numerator[]   = {1, 3, 2, 3, 4, 6, 8, 12};
static const int mclk_denominator[] = {1, 2, 1, 1, 1, 1, 1, 1};
#ifdef V_DEBUG
static const char * mclk_sdiv[]	    = {"1", "1.5", "2", "3", "4", "6", "8", "12"};
#endif

/*
 * find index >= idx, such that, for a given f_out,
 * 3 * f_mclk / 4 <= f_PLLOUT < 13 * f_mclk / 4
 * f_out can be f_256fs or f_opclk, currently only used for f_256fs. Can be
 * generalised for f_opclk with suitable coefficient arrays, but currently
 * the OPCLK divisor is calculated directly, not iteratively.
 */
static int wm897X_enum_mclk(unsigned int f_out, unsigned int f_mclk, unsigned int *f_pllout)
{
	int i;

	dbg_prt();

	for (i = 0; i < ARRAY_SIZE(mclk_numerator); i++) {
		unsigned int f_pllout_x4 =
			4 * f_out * mclk_numerator[i] / mclk_denominator[i];
		if (3 * f_mclk <= f_pllout_x4 && f_pllout_x4 < 13 * f_mclk) {
			*f_pllout = f_pllout_x4 / 4;

			vdbg_prt("%s%u %s%s %s%u %s\n",
				 ">>> MCLKDIV divides f_PLLOUT=", *f_pllout,
				 "Hz by ", mclk_sdiv[i],
				 "giving SYSCLK(256*Fs)=", f_out,
				 "Hz");
			
			return i;
		}
	}

	return -EINVAL;
}

/*
 * Calculate internal frequencies and dividers, according to Figure 41
 * "PLL and Clock Select Circuit" in wm8978 datasheet, June 2009, Rev. 4.4
 */
static int wm897X_configure_pll(struct snd_soc_codec *codec)
{
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	struct wm897X_pll_div pll_div;
	unsigned int f_opclk = wm897X->f_opclk;
	unsigned int f_mclk = wm897X->f_mclk;
	unsigned int f_256fs = wm897X->f_256fs;
	unsigned int f2;

	dbg_prt();

	if (!f_mclk) {
		vdbg_prt("%s\n", "MCLK is 0 Hz. Aborting !");
		return -EINVAL;
	}

	if (f_opclk) {
		unsigned int opclk_div;
		vdbg_prt("%s\n", "Calculating for OPCLK:");

		/* Cannot set up MCLK divider now, do later */
		wm897X->mclk_idx = -1;

		/*
		 * The user needs OPCLK. Choose OPCLKDIV to put
		 * 6 <= R = f2 / f1 < 13, 1 <= OPCLKDIV <= 4.
		 * f_opclk = f_mclk * prescale * R / 4 / OPCLKDIV, where
		 * prescale = 1, or prescale = 2. Prescale is calculated inside
		 * wm897X_pll_factors(). We have to select f_PLLOUT, such that
		 * f_mclk * 3 / 4 <= f_PLLOUT < f_mclk * 13 / 4. Must be
		 * f_mclk * 3 / 16 <= f_opclk < f_mclk * 13 / 4.
		 */
		if (16 * f_opclk < 3 * f_mclk || 4 * f_opclk >= 13 * f_mclk) {
			vdbg_prt("%s %u %s\n",
				 "Can't set for OPCLK = ",
				 f_opclk, " Hz. Aborting !");
			return -EINVAL;
		}

		if (4 * f_opclk < 3 * f_mclk) {
			/* Have to use OPCLKDIV */
			opclk_div = (3 * f_mclk / 4 + f_opclk - 1) / f_opclk;
		} else {
			opclk_div = 1;
		}

		vdbg_prt("%s %d\n", "OPCLKDIV =", f_opclk);

		snd_soc_update_bits(codec, WM897X_GPIO_CONTROL, 0x030, (opclk_div - 1) << 4);

		wm897X->f_pllout = f_opclk * opclk_div;
		

	} else if (f_256fs) {
		int idx;
		vdbg_prt("%s\n", "Calculating for PLL:");

		/*
		 * Not using OPCLK, but PLL is used for the codec, choose R:
		 * 6 <= R = f2 / f1 < 13, to put 1 <= MCLKDIV <= 12.
		 * f_256fs = f_mclk * prescale * R / 4 / MCLKDIV, where
		 * prescale = 1, or prescale = 2. Prescale is calculated inside
		 * wm897X_pll_factors(). We have to select f_PLLOUT, such that
		 * f_mclk * 3 / 4 <= f_PLLOUT < f_mclk * 13 / 4. Must be
		 * f_mclk * 3 / 48 <= f_256fs < f_mclk * 13 / 4. This means MCLK
		 * must be 3.781MHz <= f_MCLK <= 32.768MHz
		 */
		idx = wm897X_enum_mclk(f_256fs, f_mclk, &wm897X->f_pllout);
		if (idx < 0) {
			vdbg_prt("%s\n",
			  "Can't set MCLKDIV to meet PLL restrictions. Aborting!");
			return idx;
		}

		wm897X->mclk_idx = idx;

		/* GPIO1 into default mode as input - before configuring PLL */
		snd_soc_update_bits(codec, WM897X_GPIO_CONTROL, 0x007, 0x000);
		
	} else {
		vdbg_prt("%s\n", "Requested clk src unknown. Aborting!");
		return -EINVAL;
	}

	f2 = wm897X->f_pllout * 4;

	vdbg_prt("%s%u %s%u %s%u %s\n",
			 "f_MCLK=", wm897X->f_mclk,
			 "Hz, f_f2=", f2,
			 "Hz, f_PLLOUT=", wm897X->f_pllout, "Hz");

	wm897X_pll_factors(codec, &pll_div, f2, wm897X->f_mclk);

	vdbg_prt("%s0x%X %s0x%X %s%s\n",
			 "PLL conf: N=",pll_div.n,
			 ", K=", pll_div.k,
			 ". f/2 PreScaler=", (pll_div.div2 == 0) ? "Off" : "On");

	/* Reconfigure only if settings have changed */
	if (memcmp(&pll_div, &wm897X->pll_div, sizeof(pll_div))) {
		
		/* Turn PLL off for configuration... */
		vdbg_prt("%s\n", "PLL setting changed: Turning PLL Off");
		snd_soc_update_bits(codec, WM897X_POWER_MANAGEMENT_1, 0x020, 0x000);

		snd_soc_write(codec, WM897X_PLL_N, (pll_div.div2 << 4) | pll_div.n);
		snd_soc_write(codec, WM897X_PLL_K1, pll_div.k >> 18);
		snd_soc_write(codec, WM897X_PLL_K2, (pll_div.k >> 9) & 0x1FF);
		snd_soc_write(codec, WM897X_PLL_K3, pll_div.k & 0x1FF);

		wm897X->pll_div = pll_div;
	}

	/* ...and on again */
	vdbg_prt("%s\n", "Turning PLL On");

	snd_soc_update_bits(codec, WM897X_POWER_MANAGEMENT_1, 0x020, 0x020);

	if (f_opclk) {
		/* Output PLL (OPCLK) to GPIO1 */
		snd_soc_update_bits(codec, WM897X_GPIO_CONTROL, 0x007, 0x004);
	}

	return 0;
}

/*****************************************************************************/

static void wm897X_regulator_enable(struct wm897X_priv *wm897X, int on)
{
	if (on)	{
		if (!wm897X->is_regulator_actived) {
			vdbg_prt("%s\n", "Turning AVDD On");
			regulator_enable(wm897X->avdd);
			wm897X->is_regulator_actived = 1;
		}
	} else {
		if (wm897X->is_regulator_actived) {
			vdbg_prt("%s\n", "Turning AVDD Off");
			regulator_disable(wm897X->avdd);
			wm897X->is_regulator_actived = 0;		
		}
	}
}

/*
 * Configure wm897X clock dividers (BCLK)
 */
static int wm897X_dai_set_clkdiv(struct snd_soc_dai *codec_dai, int div_id, int div)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	unsigned int  bclkdiv_by;
	int ret = 0;

	dbg_prt();

	switch (div_id) {
	case WM897X_OPCLKDIV:

		vdbg_prt("%s 0x%X\n", "Requested OPCLKDIV, val =", div);
		
		wm897X->f_opclk = div;

		/* A.M. Note: Since we don't use OPCLK in DSPG products.
		**      Consider to change this code accordingly !
		*/
		
		if (wm897X->f_mclk) {
			/*
			 * We know the MCLK frequency, the user has requested
			 * OPCLK, configure the PLL based on that and start it
			 * and OPCLK immediately. We will configure PLL to match
			 * user-requested OPCLK frquency as good as possible.
			 * In fact, it is likely, that matching the sampling
			 * rate, when it becomes known, is more important, and
			 * we will not be reconfiguring PLL then, because we
			 * must not interrupt OPCLK. But it should be fine,
			 * because typically the user will request OPCLK to run
			 * at 256fs or 512fs, and for these cases we will also
			 * find an exact MCLK divider configuration - it will
			 * be equal to or double the OPCLK divisor.
			 */
			vdbg_prt("%s\n", "Setting PLL for OPCLK");
			ret = wm897X_configure_pll(codec);
		}
		break;
		
	case WM897X_BCLKDIV:

		vdbg_prt("%s 0x%X\n", "Requested BCLKDIV, val =", div);

		if (div & ~0x01C) {
			/* ? BCLKDIV = '000' is valid - :1 */
			return -EINVAL;	
										
		}

		bclkdiv_by = 0x1 << (div >> 2); // 2^n; where n = bits 4:2 of "div"
		vdbg_prt("%s %u %s%u%s\n",
			   "BCLKDIV divides SYSCLK(256*Fs) by", bclkdiv_by,
			   "giving BCLK=", 256/bclkdiv_by, "*Fs Hz");
		snd_soc_update_bits(codec, WM897X_CLOCKING, 0x01C, div);
		break;
		
	default:
		vdbg_prt("%s %u %s\n", "Unkown divider ID", div_id, "Aborting!");
		return -EINVAL;
	}

	return ret;
}

/*
 * Keep or Set wm897X PLL configuration for internal 256*Fs clock.
 * (PLL may be turned-off, if bypassed).
 */
static int wm897X_dai_set_sysclk( struct snd_soc_dai *codec_dai, int clk_id,
				  unsigned int freq, int dir )
{
	struct snd_soc_codec *codec = codec_dai->codec;
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	int ret = 0;

	dbg_prt();
	
	vdbg_prt("%s %s %s%u %s\n", "Requested CODEC clk Src:",
			 (clk_id == WM897X_PLL) ? "PLL" : "MCLK",
			 ". MCLK=", freq, "Hz");

	if (freq) {
		wm897X->f_mclk = freq;

		/* Even if MCLK is used for system clock, might have to drive OPCLK */
		if (wm897X->f_opclk) {
			vdbg_prt("%s\n", "Setting PLL for OPCLK !");
			ret = wm897X_configure_pll(codec);
		}

		/* Our sysclk is fixed to 256 * fs, will configure in .hw_params()  */
		if (!ret) {
			vdbg_prt("%s\n", "Will set PLL when hw_params() gets called");
			wm897X->sysclk = clk_id;
		}
	}

	/* Turn PLL Off, if Not used */
	if (wm897X->sysclk == WM897X_PLL && (!freq || clk_id == WM897X_MCLK)) {

		vdbg_prt("%s/n", "PLL unused: Bypassing and running directly from MCLK !");

		/* Clock CODEC directly from MCLK */
		snd_soc_update_bits(codec, WM897X_CLOCKING, 0x100, 0x000);

		/* GPIO1 into default mode as input - before configuring PLL */
		snd_soc_update_bits(codec, WM897X_GPIO_CONTROL, 7, 0x000);

		/* Turn off PLL */
		snd_soc_update_bits(codec, WM897X_POWER_MANAGEMENT_1, 0x020, 0x000);
		vdbg_prt("%s/n", "PLL turned Off");
		
		wm897X->sysclk = WM897X_MCLK;
		wm897X->f_pllout = 0;
		wm897X->f_opclk = 0;
	}

	return ret;
}

static int wm897X_dai_set_pll(struct snd_soc_dai *codec_dai, int pll_id, int source,
	unsigned int freq_in, unsigned int freq_out)
{
	unsigned int f_sel, diff, diff_best = INT_MAX;
	int i, best = 0;
	
	struct snd_soc_codec *codec = codec_dai->codec;
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	
	dbg_prt("%s\n", "configuring wolfson to work with or without pll ---->");
	
	if (!wm897X->f_mclk) {
		dbg_prt("%s\n", "mclk hadn't been set!'");
		return -EINVAL;
	}
		
	
	
	if (wm897X->sysclk == WM897X_MCLK) {
		vdbg_prt("%s\n", "Requested CODEC clk src is MCLK (PLL bypass)");
		wm897X->mclk_idx = -1;
		f_sel = wm897X->f_mclk;

	} else {
		int ret;
		vdbg_prt("%s\n", "Requested CODEC clk src is PLL - set & activate it Now");

		/* Configure and Activate PLL */
		ret = wm897X_configure_pll(codec);
		if (ret < 0) {
			vdbg_prt("%s\n", "PLL configuration failed. Aborting !");
			return ret;
		}
		f_sel = wm897X->f_pllout;
	}

	if (wm897X->mclk_idx < 0) {
		/* Either MCLK is used directly, or OPCLK is used */
		if ( (f_sel < wm897X->f_256fs) ||		// Can't divide into a higher val ...
			 (f_sel > 12 * wm897X->f_256fs) ) {	// Can't divide by more than 12 ...
			vdbg_prt("%s\n", "Can't set MCLKDIV. Change OPCLK (if used) or MCLK. Aborting !");
			return -EINVAL;
		}

		for (i = 0; i < ARRAY_SIZE(mclk_numerator); i++) {
			diff = abs(wm897X->f_256fs * 3 -
				   f_sel * 3 * mclk_denominator[i] / mclk_numerator[i]);

			if (diff < diff_best) {
				diff_best = diff;
				best = i;
			}

			if (!diff) {
				break;
			}
		}
		
	} else {
		/* OPCLK not used, codec driven by PLL */
		best = wm897X->mclk_idx;
		diff = 0;
	}

	if (diff) {
		dev_warn(codec->dev, "Imprecise sampling rate: %uHz%s\n",
			f_sel * mclk_denominator[best] / mclk_numerator[best] / 256,
			wm897X->sysclk == WM897X_MCLK ?
			", consider using PLL" : "");
	}

	vdbg_prt("%s 0x%X %s %u %s %d\n", "pcm fmt", wm897X->format,
			 ", rate", wm897X->rate,
			 ", MCLK divisor #", best);

	/* MCLK divisor mask = 0x0E0 */
	snd_soc_update_bits(codec, WM897X_CLOCKING, 0x0E0, best << 5);

	snd_soc_write(codec, WM897X_AUDIO_INTERFACE, wm897X->iface_ctl);

	snd_soc_write(codec, WM897X_ADDITIONAL_CONTROL, wm897X->add_ctl);

	// if (wm897X->sysclk != current_clk_id) { - zivh test not relevant
	if (wm897X->sysclk == WM897X_PLL) {
		/* Clock CODEC from PLL */
		snd_soc_update_bits(codec, WM897X_CLOCKING, 0x100, 0x100);
	} else {
		/* Clock CODEC from MCLK (Bypass PLL) */
		snd_soc_update_bits(codec, WM897X_CLOCKING, 0x100, 0x000);
	}
	//}

	return 0;
}


/* Set TDM physical-layer protocol */
static int wm897X_dai_set_fmt(struct snd_soc_dai *codec_dai, unsigned int fmt)
{
	struct snd_soc_codec *codec = codec_dai->codec;
	/*
	 * BCLK(sclk) clock polarity mask: 0x100.
	 * LRC(fsync) clock polarity mask: 0x080.
	 * Data Format mask:        0x018.
	 */
	u16 iface = snd_soc_read(codec, WM897X_AUDIO_INTERFACE) & ~0x198;
	u16 clk = snd_soc_read(codec, WM897X_CLOCKING);

	dbg_prt();

	/* master/slave audio interface */
	switch (fmt & SND_SOC_DAIFMT_MASTER_MASK) {
	case SND_SOC_DAIFMT_CBM_CFM:
		vdbg_prt("%s\n", "Wolfson is TDM Master");
		clk |= 0x001;	// BCLK & LRC Master - MS = '1'
		g_is_codec_master = 1;
		break;
	case SND_SOC_DAIFMT_CBS_CFS:
		vdbg_prt("%s\n", "Wolfson is TDM Slave");
		clk &= ~0x001;	// BCLK & LRC Slave  - MS = '0'
		g_is_codec_master = 0;
		break;
	default:
		return -EINVAL;
	}

	/* interface format */
	switch (fmt & SND_SOC_DAIFMT_FORMAT_MASK) {
	case SND_SOC_DAIFMT_I2S:
		vdbg_prt("%s\n", "TDM format is I2S");
		iface |= 0x010;		// FMT = '10'	--> I2S
		break;
	case SND_SOC_DAIFMT_RIGHT_J:
		vdbg_prt("%s\n", "TDM format is Right-Justified");
		break;			// FMT = '00'	--> Right justified
	case SND_SOC_DAIFMT_LEFT_J:
		vdbg_prt("%s\n", "TDM format is Left-Justified");
		iface |= 0x008;		// FMT = '01'	--> Left justified
		break;
	case SND_SOC_DAIFMT_DSP_A:
		vdbg_prt("%s\n", "TDM format is PCM/DSP");
		iface |= 0x018;		// FMT = '11'	--> PCM/DSP (mode A|B by LRP = 0|1)
		break;
	default:
		return -EINVAL;
	}

	/* clocks polarity */
	switch (fmt & SND_SOC_DAIFMT_INV_MASK) {
	case SND_SOC_DAIFMT_NB_NF:
		vdbg_prt("%s\n", "TDM clocks polarity: normal BCLK(sclk); normal LRC(FSync)");
		break;			// BCP = '0', LRP = '0' --> for i2s: normal BCLK & LRC
 	case SND_SOC_DAIFMT_IB_IF:
		vdbg_prt("%s\n", "TDM clocks polarity: inverted BCLK(sclk); inverted LRC(FSync)");
		iface |= 0x180;		// BCP = '1', LRP = '1' --> for i2s: inverted BCLK & LRC
		break;
	case SND_SOC_DAIFMT_IB_NF:
		vdbg_prt("%s\n", "TDM clocks polarity: inverted BCLK(sclk); normal LRC(FSync)");
		iface |= 0x100;		// BCP = '1', LRP = '0' --> for i2s: inverted BCLK, normal LRC
		break;
	case SND_SOC_DAIFMT_NB_IF:
		vdbg_prt("%s\n", "TDM clocks polarity: normal BCLK(sclk); inverted LRC(FSync)");
		iface |= 0x080;		// BCP = '0', LRP = '1' --> for i2s: normal BCLK, inverted LRC
		break;
	default:
		return -EINVAL;
	}

	snd_soc_write(codec, WM897X_AUDIO_INTERFACE, iface);
	snd_soc_write(codec, WM897X_CLOCKING, clk);

	return 0;
}

/* Set bit-witdh and sample-rate (fs) */
static int wm897X_dai_hw_params(struct snd_pcm_substream *substream,
			    struct snd_pcm_hw_params *params,
			    struct snd_soc_dai *dai)
{
	u16  clking;

	enum wm897X_sysclk_src current_clk_id;

	struct snd_soc_pcm_runtime *rtd = substream->private_data;
	struct snd_soc_codec *codec = rtd->codec;
	//struct snd_soc_card *card = codec->card;
	//struct wm897X_setup_data *setup_data = snd_soc_card_get_drvdata(card);

	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);

	dbg_prt("%s\n", DIR_STR(substream->stream));

	/* Word length mask = 0x060 */
	wm897X->iface_ctl = snd_soc_read(codec, WM897X_AUDIO_INTERFACE) & ~0x060;	// Reg 4's WL <-- '00'

	/* Sampling rate mask = 0x00E (for filters) */
	wm897X->add_ctl = snd_soc_read(codec, WM897X_ADDITIONAL_CONTROL) & ~0x00E;	// Reg 7's SR <-- '000'

	clking = snd_soc_read(codec, WM897X_CLOCKING);				// Reg 6's CLKSEL
	current_clk_id = clking & 0x100 ? WM897X_PLL : WM897X_MCLK;

	vdbg_prt("%s %u %s\n", "Start setting for Sample-Rate: ",
			params_rate(params), "Hz");

	/* Bit width */

	/* Note: 
	 *   In Master mode, the Wolfson chip supports bit-width
	 *   ("word length") of 16, 20, 24 and 32 bits per channel Only.
	 *   In Slave mode, it can "absorb" more BCLKs-to-the-right,
	 *   than the configured bit-width (WL field of reg 4).
	 *
	 *   We make use of this feature, when dmw-96 is the TDM Master,
	 *   by setting 16 to 20 (!) bits in a time-slot, in order to get
	 *   better sclk (BCLK) accuracy - but we leave the Wolfson
	 *   configuration at 16 bits.
	 */
	 
	switch (wm897X->format = params_format(params)) {
	case SNDRV_PCM_FORMAT_S16_LE:
		vdbg_prt("%s\n", "PCM sample-length (vs. ALSA) = 16 bits; LE");
		break;			// WL = '00' --> 16 bits
	case SNDRV_PCM_FORMAT_S20_3LE:
		vdbg_prt("%s\n", "PCM sample-length (vs. ALSA) = 20 bits; LE");
		wm897X->iface_ctl |= 0x020;	// WL = '01' --> 20 bits
		break;
	case SNDRV_PCM_FORMAT_S24_LE:
		vdbg_prt("%s\n", "PCM sample-length (vs. ALSA) = 24 bits; LE");
		wm897X->iface_ctl |= 0x040;	// WL = '10' --> 24 bits
		break;
	case SNDRV_PCM_FORMAT_S32_LE:
		vdbg_prt("%s\n", "PCM sample-length (vs. ALSA) = 32 bits; LE");
		wm897X->iface_ctl |= 0x060;	// WL = '11' --> 32 bits
		break;
	}

	/* Filter coefficient (per sample-rate) */
	switch (wm897X->rate = params_rate(params)) {
	case 8000:
		wm897X->add_ctl |= (0x5 << 1);	// SR = '101'
		break;
	case 11025:
	case 12000:
		wm897X->add_ctl |= (0x4 << 1);	// SR = '100'
		break;
	case 16000:
		wm897X->add_ctl |= (0x3 << 1);	// SR = '011'
		break;
	case 22050:
	case 24000:
		wm897X->add_ctl |= (0x2 << 1);	// SR = '010'
		break;
	case 32000:
		wm897X->add_ctl |= (0x1 << 1);	// SR = '001'
		break;
	case 44100:
	case 48000:			// SR = '000'
		break;					
	}

	/* Sampling rate is known now, can configure the MCLK divider */
	wm897X->f_256fs = params_rate(params) * 256;
	vdbg_prt("%s %u\n", "256*Fs =", wm897X->f_256fs);


	return 0;
}

static int wm897X_dai_hw_free(struct snd_pcm_substream *substream,
	struct snd_soc_dai *cpu_dai)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}

static int wm897X_dai_prepare(struct snd_pcm_substream *substream,
	struct snd_soc_dai *dai)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}	

static int wm897X_dai_trigger(struct snd_pcm_substream *substream, int cmd,
		struct snd_soc_dai *dai)
{
	dbg_prt("%s %s\n",
		DIR_STR(substream->stream),
		"Currently doing nothing ---->");

	return 0;
}	

static int wm897X_dai_mute(struct snd_soc_dai *dai, int mute)
{
	struct snd_soc_codec *codec = dai->codec;

	dbg_prt("%s %s\n", "set DACs Soft mute", (mute == 0) ? "Off" : "On");

	/* Use reg 10 SOFTMUTE for gradual mute/unmute
	 * (instead of reg 11/12 DACVOLL/DACVOLLR = 0x00)
	 */
	if (mute) {
		snd_soc_update_bits(codec, WM897X_DAC_CONTROL, 0x040, 0x040);
	} else {
		snd_soc_update_bits(codec, WM897X_DAC_CONTROL, 0x040, 0x000);
	}

	return 0;
}

static int wm897X_dai_startup(struct snd_pcm_substream *substream,
		struct snd_soc_dai *dai)
{
	struct snd_soc_codec *codec = dai->codec;
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);

	dbg_prt("%s\n", DIR_STR(substream->stream));

	 /* set_bias_level:
	  * in case there is a playback and the lcd is off we first check what is our bias level - in case it is off we will switch to standby
	  * this code is solving an issue of not using the kernel suspend/resume code but trying to playback when lcd is closed
	  */
	mutex_lock(&wm897X->bias_mutex);

	if 	(codec->dapm.bias_level == SND_SOC_BIAS_OFF){
		dbg_prt("%s\n", "bias_level is OFF - switching to standby");
		wm897X_set_bias_level(codec,SND_SOC_BIAS_STANDBY);
	}

	mutex_unlock(&wm897X->bias_mutex);

	//g_opencnt++;
	atomic_inc(&g_opencnt);
		
	return 0;
}

static void wm897X_dai_shutdown(struct snd_pcm_substream *substream,
		struct snd_soc_dai *dai)
{
	dbg_prt("%s\n", DIR_STR(substream->stream));

	//if (--g_opencnt == 0) {
	if (atomic_dec_return(&g_opencnt) == 0) {

		/* Disable Wolfson internal PLL */
		vdbg_prt("%s %s%s\n",
			 "codec-dai", dai->name, ": Disabling WM PLL");
		snd_soc_update_bits(dai->codec, WM897X_POWER_MANAGEMENT_1, 0x020, 0x000);
	}
}	
	
/*****************************************************************************/

static void wm897X_set_slow_clock(struct snd_soc_codec *codec, uint en)
{
	u16 reg_val;

	dbg_prt();

	reg_val = snd_soc_read(codec, WM897X_ADDITIONAL_CONTROL);
	if (en) {
		reg_val |= 0x001;	/* '1' --> SLOWCLKEN */
		vdbg_prt("%s\n", "ZC slow-clk Enabled");
	} else {

		reg_val &= ~0x001;	/* '0' --> SLOWCLKEN */
		vdbg_prt("%s\n", "ZC slow-clk Disabled");
	}
	snd_soc_write(codec, WM897X_ADDITIONAL_CONTROL, reg_val);
}

/* set_bias_level function is now being accessed from two different contexts
 * therefor we are adding a mutex to protect this access and also handling the 
 * state changes localy and not just via the dapm_reg_event
 */

static int wm897X_set_bias_level_locked(struct snd_soc_codec *codec,
					enum snd_soc_bias_level level)
{
	int ret;

	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);

	dbg_prt();

	/* lock mutex */
	mutex_lock(&wm897X->bias_mutex);

	ret = wm897X_set_bias_level(codec,level);

	/*unlock mutex */
	mutex_unlock(&wm897X->bias_mutex);

	return ret;

}
static int wm897X_set_bias_level(struct snd_soc_codec *codec,
					enum snd_soc_bias_level level)
{
	u16 reg1;
	bool use_tieoff = false;
	struct snd_soc_card *card = codec->card;
	struct wm897X_setup_data *setup_data;
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	int i;
	u16 *cache = codec->reg_cache;
	int ret;

	dbg_prt();

	vdbg_prt("%s %d %s %d %s %d\n", "set_bias_level: current level:",codec->dapm.bias_level,"requested level:", level, "idle_bias_off", codec->dapm.idle_bias_off);

	/* Note: Setting VMIDSEL = '00' also turns the Wolfson PLL OFF !!!
	 *       ---------------------------------------------------------
	 */

	reg1 = snd_soc_read(codec, WM897X_POWER_MANAGEMENT_1);
	vdbg_prt("%s 0x%X\n", "Initial Reg1:", reg1);

	reg1 &= ~0x003; /* VMIDSEL <-- '00' (open circuit) */

	switch (level) {

	case SND_SOC_BIAS_ON:

		vdbg_prt("%s\n", "Bias ON: Setting VMID 75K");

		reg1 |= 0x001;  /* VMID 75k */
		snd_soc_write(codec, WM897X_POWER_MANAGEMENT_1, reg1); /* VMIDSEL <-- '01' */

		vdbg_prt("%s 0x%X\n", "Bias ON: Reg1 after:", reg1);

		/* Change to new state */
		codec->dapm.bias_level = level;
		break;

	case SND_SOC_BIAS_PREPARE:
		vdbg_prt("%s\n", "Bias PREPARE: VMID 75K, Enable SlowClok");

		reg1 |= 0x001;  /* VMID 75k */
		snd_soc_write(codec, WM897X_POWER_MANAGEMENT_1, reg1); /* VMIDSEL <-- '01' */
		/* Enable slow-clock (used for volume-update on zero-cross Timeout) */
		wm897X_set_slow_clock(codec, 1);

		vdbg_prt("%s 0x%X\n", "Bias PREPARE: Reg1 after:", reg1);

		/* Change to new state */
		codec->dapm.bias_level = level;
		break;

	case SND_SOC_BIAS_STANDBY:
		if (codec->dapm.bias_level == SND_SOC_BIAS_OFF) {

			vdbg_prt("%s\n", "Bias STANDBY: OFF ---> STANDBY !");		
			/* before starting the wolfson enable our vcca and wait - remove the wait later */
			wm897X_regulator_enable(wm897X,1);
			msleep(5);

	/* Sync codec's regs with the reg-cache (skip the Reset register!).
	 * This may cause pop/click noise
	 */
			vdbg_prt("%s\n", "Bias STANDBY: writing reset values to wolfson !");
			/* Reset the codec */
			ret = snd_soc_write(codec, WM897X_RESET, 0);
			if (ret < 0) {
				dev_err(codec->dev, "Failed to issue reset\n");
				return ret;
			}

			for (i = WM897X_RESET + 1; i < ARRAY_SIZE(wm897X_reg_reset_val); i++) {
				if (cache[i] != wm897X_reg_reset_val[i]) {
#ifdef _ALTERNATE_WRITE_FUNCT
					wm897X_uncond_reg_write(codec, i, cache[i]);
#else
					snd_soc_write(codec, i, cache[i]);
#endif
				}
			}
			/* OFF --> STANDBY transition: Impose power-up sequence */
		
			setup_data = snd_soc_card_get_drvdata(card);
			if (setup_data != NULL) {
				use_tieoff = setup_data->use_tie_off;
			}

			vdbg_prt("%s %s\n", "Outputs Tie-Off",
				(use_tieoff) ? "used" : "unused");

			/* Step 1: */
			/* ------  */

			if (use_tieoff) {
				/* Note:
				*  DC Blocking (AC Coupling) Capactiors charge,
				*  when tie-off is enabled, towards Vmid, via
				*  the Speakers/Earpiece/Earphones output devices.
				*/

				/* Limit the charge current of outputs DC-Blocking
				 * serial capacitors (if exist), by selecting
				 * high-value tie-off serial resistors:
				 * VROI <-- '1' : 30k
				 */
				vdbg_prt("%s\n",
					"Bias STANDBY step 1: VROI 30K. Enable Tie-Off");
				snd_soc_update_bits(codec,
					WM897X_OUTPUT_CONTROL, 0x001, 0x001);

				/* Enable Tie-Off */
				reg1 |= 0x004;  /* bit2 BUFIOEN <-- '1' */
			}

			/* Keep Bias Disabled  */
			reg1 &= ~0x008; /* bit3 BIASEN  <-- '0' */

#if 0 			/* i think this wrong sequence - we want the highest value to have slow stabilization and less fisability for clicks */
			/* Set VMID with Smallest impedance
			 * (for fast inputs/outputs ramp up towards Vmid)
			 */
			reg1 |= 0x003;	/* VMID 5k */
#else 
			reg1 |= 0x002;	/* VMID 300k */
#endif 
			vdbg_prt("%s\n",
				"Bias STANDBY step 1: VMID 5K. Bias Off");
			snd_soc_write(codec,
				WM897X_POWER_MANAGEMENT_1, reg1); /* VMIDSEL <-- '11' */

			vdbg_prt("%s 0x%X\n",
				"Bias STANDBY step 1: Reg1 after:", reg1);

			/* Wait VMID ramp-up (and serial caps charge) */
			//mdelay(100); - used to be			
			mdelay(500);

			/* Step 2: */
			/* ------  */

			/* Enable Bias */
			vdbg_prt("%s\n", "Bias STANDBY step 2: Bias On");
			reg1 |= 0x008; /* bit3 BIASEN  <-- '1' */
			snd_soc_write(codec, WM897X_POWER_MANAGEMENT_1, reg1);

			vdbg_prt("%s 0x%X\n",
				"Bias STANDBY step 2: Reg1 after:", reg1);

			if (use_tieoff) {
				/* Select low-value tie-off serial resistors:
				 * VROI <-- '0' : 1k
				 */
				vdbg_prt("%s\n",
					"Bias STANDBY step 2: VROI 1K");
				snd_soc_update_bits(codec,
					WM897X_OUTPUT_CONTROL, 0x001, 0x000);

				mdelay(20);
			}

			reg1 &= ~0x003;
		}

		/* Change to new state */
		codec->dapm.bias_level = level;
#if 0
		/* Set VMID with midrange impedance */
		vdbg_prt("%s\n", "Bias STANDBY: VMID 75K");
		//reg1 |= 0x0002;/* VMID 300k */		
		reg1 |= 0x0001;	/* VMID 75k */
		snd_soc_write(codec, WM897X_POWER_MANAGEMENT_1, reg1); /* VMIDSEL <-- '01' */

		vdbg_prt("%s 0x%X\n", "Bias STANDBY: Reg1 after:", reg1);
#endif
		break;

	case SND_SOC_BIAS_OFF:
		/* if we are already in standby then shutdown everyting and set the idle_bias_off bit */
		if (codec->dapm.bias_level == SND_SOC_BIAS_STANDBY) {
			vdbg_prt("%s\n\t\t %s\n",
				"Bias OFF: Nuliffy Regs 2,3, most of reg 1",
				"WM PLL implicitly disabled. Disable SlowClok");

			/* Note:
			 * 1. PLL gets Disabled.
			 * 2. All outputs get Disabled:
			 *    - ROUT1EN, LOUT1EN: Reg2 - 8:7
			 *    - LOUT2EN, ROUT2EN: Reg3 - 6:5
			 *    - OUT4EN, OUT3EN:   Reg3 - 8:7
			 * 3. VMID placed as open loop
			 */

			//snd_soc_update_bits(codec, WM897X_POWER_MANAGEMENT_1, 0x1DF, 0x000);
			snd_soc_update_bits(codec, WM897X_POWER_MANAGEMENT_1, 0x1FF, 0x000);
			snd_soc_write(codec, WM897X_POWER_MANAGEMENT_2, 0x000);
			snd_soc_write(codec, WM897X_POWER_MANAGEMENT_3, 0x000);

			/* Disable slow-clock */
			wm897X_set_slow_clock(codec, 0);
			
			/* disable regular - cuts our VCC */
			wm897X_regulator_enable(wm897X,0);

			vdbg_prt("%s\n", "Bias OFF: completed");

			/* mark our mode as bias off */			
			codec->dapm.idle_bias_off = 1;

			/* Change to new state */
			codec->dapm.bias_level = level;

		/* we are still active/prepared so we will mark to the alsa core that when closing we want to be shuted down */
		} else {
			vdbg_prt("%s\n",
				"Bias OFF: still active, mark idle_bias_off to dapm ");
			codec->dapm.idle_bias_off = 1;
			/* we must not! change the level in this case */
		}
	
		break;
	}

	return 0;
}

/* in early suspend:
 * we will always call to set bias to off - the bias function itself will check if reaching to off mode is acceptable
 */
static void wm897X_early_suspend(struct early_suspend *es) 
{

	struct wm897X_priv* wm897X = container_of(es, struct wm897X_priv,wm897X_early_suspend );

	mutex_lock(&wm897X->bias_mutex);
	
	wm897X_set_bias_level(wm897X->codec, SND_SOC_BIAS_OFF);

	mutex_unlock(&wm897X->bias_mutex);
	
}

/* in late resume
 * we will always call to set the bias to standby - the bias function itself will check if the device can go to standby is
 * already in standby so no action should be taken
 */
static void wm897X_late_resume(struct early_suspend *es) {

	struct wm897X_priv* wm897X = container_of(es, struct wm897X_priv, wm897X_early_suspend);
	int level;


	mutex_lock(&wm897X->bias_mutex);

	level = wm897X->codec->dapm.bias_level;

	/*only if we are not already active call set standby */
	if (level == SND_SOC_BIAS_OFF){
		vdbg_prt("%s\n","late resume: the bias_level is off we can go to standby");
		wm897X_set_bias_level(wm897X->codec, SND_SOC_BIAS_STANDBY);
	}

	vdbg_prt("%s\n","late resume: the bias_level is not off therefor all we can do is just set the ");

	wm897X->codec->dapm.idle_bias_off = 0; /* don't allow to go to deep sleep while screen is active */

	mutex_unlock(&wm897X->bias_mutex);
	
}
/*****************************************************************************/

/* Note: As TDM Master, wm897X supports 16, 20, 24 or 32 bits per "channel" Only.
 *	 As TDM Slave, it supports 16 to 32 bits consecutively.
 *
 *	The chip also supports 8-bit Linear-PCM and ALaw/uLaw-companded-PCM.
 *	This ability is Not implemented in code.
 */
#define WM897X_FORMATS (SNDRV_PCM_FMTBIT_S16_LE | SNDRV_PCM_FMTBIT_S20_3LE | \
			SNDRV_PCM_FMTBIT_S24_LE | SNDRV_PCM_FMTBIT_S32_LE)

static struct snd_soc_dai_ops wm897X_dai_ops = {
	.set_sysclk	= wm897X_dai_set_sysclk,
	.set_pll	= wm897X_dai_set_pll,
	.set_clkdiv	= wm897X_dai_set_clkdiv,	/* Mainly BCLK divider */
	.set_fmt	= wm897X_dai_set_fmt,
	.digital_mute	= wm897X_dai_mute,
	.startup	= wm897X_dai_startup,
	.shutdown	= wm897X_dai_shutdown,
	.hw_params	= wm897X_dai_hw_params,
	.hw_free	= wm897X_dai_hw_free,
	.prepare	= wm897X_dai_prepare,
	.trigger	= wm897X_dai_trigger,
};

static struct snd_soc_dai_driver wm897X_dai = {
	.name = "wm897X-hifi",
	.playback = {
		.stream_name	= "Playback",
		.channels_min	= 1,
		.channels_max	= 2,
		.rates 		= SNDRV_PCM_RATE_8000_48000,
		.formats 	= WM897X_FORMATS,
	},
	.capture = {
		.stream_name	= "Capture",
		.channels_min	= 1,
		.channels_max	= 2,
		.rates		= SNDRV_PCM_RATE_8000_48000,
		.formats	= WM897X_FORMATS,
	},
	.ops = &wm897X_dai_ops,
};

/*****************************************************************************/

static int wm897X_dev_suspend(struct snd_soc_codec *codec, pm_message_t state)
{
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	dbg_prt();

	wm897X_set_bias_level(codec, SND_SOC_BIAS_OFF);

	/* Note:
	 * The power consuming components, controlled by registers 2 & 3,
	 * are Already turned Off due to DAPM routes shutdown
	 * and bias-level set to BIAS_OFF above. As consequence of
	 * BIAS_OFF (VMIDSEL == '00'), the Wolfson PLL was silently Disabled.
	 *
	 * Here, on register 1, the Wolfson PLL gets Explicitly shut-down.
	 */

	vdbg_prt("%s\n", "Disabling VMIDSEL, BUFIOEN, BIASEN, PLLEN, BUFDCOPEN !");

	 /* Since OUT3 & OUT4 are used in DSPG products, OUT3MIXEN & OUT4MIXEN,
	 * which appear in Our DAPM routes, should be preserved when turning off
	 * register 1 components.
	 */
	//snd_soc_write(codec, WM897X_POWER_MANAGEMENT_1, 0x000);
	snd_soc_update_bits(codec, WM897X_POWER_MANAGEMENT_1, 0x13F, 0x000);

	regulator_disable(wm897X->avdd);

	return 0;
}

static int wm897X_dev_resume(struct snd_soc_codec *codec)
{
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	int i;
	u16 *cache = codec->reg_cache;

	dbg_prt();

	regulator_enable(wm897X->avdd);
	mdelay(10);

	/* Note:
	** If a register has Ever been changed since power-up, it will be re-written here.
	** This implies that under 'suspend', a possible Power-Off of the codec chip
	** (external to this driver) is being Assumed !
	*/

	/* Sync codec's regs with the reg-cache (skip the Reset register!).
	 * This may cause pop/click noise
	 */
	for (i = WM897X_RESET + 1; i < ARRAY_SIZE(wm897X_reg_reset_val); i++) {
		if (cache[i] != wm897X_reg_reset_val[i]) {
#ifdef _ALTERNATE_WRITE_FUNCT
			wm897X_uncond_reg_write(codec, i, cache[i]);
#else
			snd_soc_write(codec, i, cache[i]);
#endif
		}
	}

	wm897X_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	if (wm897X->f_pllout) {

		/* Note: See wm897X_dev_suspend() */
		/* Switch PLL on */
		snd_soc_update_bits(codec, WM897X_POWER_MANAGEMENT_1, 0x020, 0x020);

		vdbg_prt("%s\n", "PLL turned On");
	}

	return 0;
}

/*****************************************************************************/

#ifdef _ALTERNATE_WRITE_FUNCT
static int wm897X_reg_write(struct snd_soc_codec *codec,
			    unsigned int reg,
			    unsigned int value)
{
	unsigned int old_reg_val;
	unsigned int i;
	unsigned int ret = 0;
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);

#ifdef DEBUG_WM_REG_WRITE
	printk(KERN_DEBUG "Wolfson write reg %u  val 0x%3.3X\n", reg, value);
#endif

	if (reg == WM897X_RESET) {
		/* Always honour writes to the Reset regiater (regardless its cached value !): */
		ret = wm897X->orig_codec_write(codec, WM897X_RESET, value);
		if (ret == 0) {
			/* Set the reg-cache to the Reset values */
			memcpy(codec->reg_cache, wm897X_reg_reset_val, sizeof(wm897X_reg_reset_val));
			/* Set the "update" bit in all registers, that have one */
			for (i = 0; i < ARRAY_SIZE(update_reg); i++) {
				((u16 *)codec->reg_cache)[update_reg[i]] |= 0x100;
			}
#ifdef DEBUG_WM_REG_WRITE
			printk(KERN_DEBUG "Wolfson Reset !\n");
#endif

		} else {
			dev_err(codec->dev, "Wolfson reset failed: %d\n", ret);
		}

		return ret;
	}

	if (value == 0xFFFF) {
		/* Silently prevent writing a reserved or non-existing codec register */
#ifdef DEBUG_WM_REG_WRITE
		printk(KERN_DEBUG "Wolfson write reg %u Skipped\n", reg);
#endif
		return 0;
	}

	/* Check for a change vs. reg-cache */
	/* always write through */
	//old_reg_val = snd_soc_read(codec, reg);
	//if (value != old_reg_val) {
		ret = wm897X->orig_codec_write(codec, reg, value);
		if (ret != 0) {
			dev_err(codec->dev, "Wolfson write failed: %d\n", ret);
		}
				
	//} else {
#ifdef DEBUG_WM_REG_WRITE
		//printk(KERN_DEBUG "Wolfson reg %u Unchanged\n", reg);
#endif
	//}

	return ret;
}

/* Use this function only(!) when (external) power-off/power-on sequence
 * of the Wolfson chip, assumed to had been occured.
 * (e.g. "resume" from "suspend").
 * This overrides the check versus the reg-cache.
 */
static int wm897X_uncond_reg_write(struct snd_soc_codec *codec,
				   unsigned int reg,
				   unsigned int value)
{
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);

	if (reg != WM897X_RESET) {

#ifdef DEBUG_WM_REG_WRITE
		printk(KERN_DEBUG "Wolfson unconditional write reg %u  val 0x%3.3X\n", reg, value);
#endif
		return wm897X->orig_codec_write(codec, reg, value);
	}

	return 0;
}

#endif /* _ALTERNATE_WRITE_FUNCT */

/*****************************************************************************/

static int wm897X_dev_probe(struct snd_soc_codec *codec)
{
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	int ret = 0;
	int i;

	dbg_prt();

	/* save point to the codec for early_suspend usage */
	wm897X->codec = codec;

	/* init mutex */
	mutex_init(&wm897X->bias_mutex);

	wm897X->avdd = regulator_get(codec->dev, "avdd");
	if (IS_ERR(wm897X->avdd)) {
		dev_err(codec->dev, "cannot get avdd regulator!\n");
		return PTR_ERR(wm897X->avdd);
	}
	/* initialize and enable the regulator */
	wm897X->is_regulator_actived = 0;
	wm897X_regulator_enable(wm897X,1);

	mdelay(10);

	/*
	 * Set default system clock to PLL, it is more precise, this is also the
	 * default hardware setting
	 */
	wm897X->sysclk = WM897X_PLL;

	codec->control_data = wm897X->control_data;

	ret = snd_soc_codec_set_cache_io(codec, 7, 9, SND_SOC_I2C);
	if (ret < 0) {
		dev_err(codec->dev, "Failed to set cache I/O: %d\n", ret);
		return ret;
	}

#ifdef _ALTERNATE_WRITE_FUNCT
	/* Override default register-write function (set by
	 * snd_soc_codec_set_cache_io() above)
	 */
	wm897X->orig_codec_write = codec->write;
	codec->write = wm897X_reg_write;
#endif /* _ALTERNATE_WRITE_FUNCT */

	/*
	 * Set the "update" bit in all registers, that have one. This way all
	 * writes to those registers will also cause the update bit to be
	 * written.
	 */
	for (i = 0; i < ARRAY_SIZE(update_reg); i++) {
		snd_soc_update_bits(codec, update_reg[i], 0x100, 0x100);

	}

	/* Reset the codec */
	ret = snd_soc_write(codec, WM897X_RESET, 0);
	if (ret < 0) {
		dev_err(codec->dev, "Failed to issue reset\n");
		return ret;
	}

#ifdef CONFIG_HAS_EARLYSUSPEND
		wm897X->wm897X_early_suspend.suspend = wm897X_early_suspend;
		wm897X->wm897X_early_suspend.resume = wm897X_late_resume;
		wm897X->wm897X_early_suspend.level = EARLY_SUSPEND_LEVEL_DISABLE_FB;
		register_early_suspend(&wm897X->wm897X_early_suspend);
#endif

	wm897X_set_bias_level(codec, SND_SOC_BIAS_STANDBY);

	ret = snd_soc_add_controls(codec, wm897X_snd_controls,
				ARRAY_SIZE(wm897X_snd_controls));
	if (ret < 0) {
		dev_err(codec->dev, "failed to add Controls: %d\n", ret);
		return ret;
	}

	ret = wm897X_add_widgets(codec);
	if (ret < 0) {
		dev_err(codec->dev, "failed to add Widgets or Routes: %d\n", ret);
		return ret;
	}

	return 0;
}

static int wm897X_dev_remove(struct snd_soc_codec *codec)
{
	struct wm897X_priv *wm897X = snd_soc_codec_get_drvdata(codec);
	dbg_prt();

	wm897X_set_bias_level(codec, SND_SOC_BIAS_OFF);
	regulator_disable(wm897X->avdd);
	regulator_put(wm897X->avdd);
	unregister_early_suspend(&wm897X->wm897X_early_suspend);

	return 0;
}

static struct snd_soc_codec_driver soc_codec_dev_wm897X = {
	.probe =	wm897X_dev_probe,
	.remove =	wm897X_dev_remove,
	.suspend =	wm897X_dev_suspend,
	.resume =	wm897X_dev_resume,
	.set_bias_level	= wm897X_set_bias_level_locked,
	.reg_cache_size	= ARRAY_SIZE(wm897X_reg_reset_val),
	.reg_word_size = sizeof(u16),
	.reg_cache_default = wm897X_reg_reset_val,
};

/*****************************************************************************/

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)

static __devinit int wm897X_i2c_probe(struct i2c_client *i2c,
					const struct i2c_device_id *id)
{
	struct wm897X_priv *wm897X;
	int ret;

	dbg_prt();

	wm897X = kzalloc(sizeof(struct wm897X_priv), GFP_KERNEL);
	if (wm897X == NULL) {
		vdbg_prt("%s\n", "Allocating codec priv data Failed");
		return -ENOMEM;
	}

	i2c_set_clientdata(i2c, wm897X);

	wm897X->control_data = i2c;

	vdbg_prt("%s\n", "Registering asoc Codec driver and Dai");
	ret = snd_soc_register_codec(&i2c->dev,
			&soc_codec_dev_wm897X, &wm897X_dai, 1);
	if (ret < 0) {
		vdbg_prt("%s\n", "Asoc codec driver and dai registration Failed !!!");
		kfree(wm897X);
	}

	return ret;
}

static __devexit int wm897X_i2c_remove(struct i2c_client *client)
{
	dbg_prt();

	vdbg_prt("%s\n", "Unregistering asoc Codec driver");

	snd_soc_unregister_codec(&client->dev);
	kfree(i2c_get_clientdata(client));  	

	return 0;
}

static const struct i2c_device_id wm897X_i2c_id[] = {
	{ "wm897X", 0 },
	{ }
};
MODULE_DEVICE_TABLE(i2c, wm897X_i2c_id);

static struct i2c_driver wm897X_i2c_driver = {
	.driver = {
		.name = "wm897X",
		.owner = THIS_MODULE,
	},
	.probe =    wm897X_i2c_probe,
	.remove =   __devexit_p(wm897X_i2c_remove),
	.id_table = wm897X_i2c_id,
};

#endif /* CONFIG_I2C || CONFIG_I2C_MODULE */

/*****************************************************************************/

static int __init wm897X_mod_Init(void)
{
	int ret = 0;

	dbg_prt();

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)
	ret = i2c_add_driver(&wm897X_i2c_driver);
	if (ret != 0) {
		printk(KERN_ERR "Failed to register wm897X I2C driver: %d\n", ret);
	}
#endif

	return ret;
	
}
module_init(wm897X_mod_Init);


static void __exit wm897X_mod_Exit(void)
{
	dbg_prt();

#if defined(CONFIG_I2C) || defined(CONFIG_I2C_MODULE)

	i2c_del_driver(&wm897X_i2c_driver);
#endif

}
module_exit(wm897X_mod_Exit);


MODULE_DESCRIPTION("ASoC wm897X codec driver");
MODULE_AUTHOR("Avi Miller <Avi.Miller@dspg.com>");
MODULE_LICENSE("GPL");

